declare module 'bcrypt' {
  /**
   * Generate a salt
   * @param rounds Number of rounds to use, defaults to 10 if omitted
   * @param callback Callback receiving the error, if any, and the generated salt
   */
  export function genSalt(rounds?: number, callback?: (err: Error | null, salt: string) => void): Promise<string>;

  /**
   * Generate a salt sync
   * @param rounds Number of rounds to use, defaults to 10 if omitted
   */
  export function genSaltSync(rounds?: number): string;

  /**
   * Hash data using a salt
   * @param data The data to be hashed
   * @param saltOrRounds The salt to be used to hash the password. If specified as a number then a
   * salt will be generated with the specified number of rounds and used
   * @param callback Callback receiving the error, if any, and the resultant hash
   */
  export function hash(data: string, saltOrRounds: string | number, callback?: (err: Error | null, encrypted: string) => void): Promise<string>;

  /**
   * Hash data using a salt sync
   * @param data The data to be hashed
   * @param saltOrRounds The salt to be used to hash the password. If specified as a number then a
   * salt will be generated with the specified number of rounds and used
   */
  export function hashSync(data: string, saltOrRounds: string | number): string;

  /**
   * Compare data with hash
   * @param data The data to be compared
   * @param encrypted The data to be compared to
   * @param callback Callback receiving the error, if any, and the result
   */
  export function compare(data: string, encrypted: string, callback?: (err: Error | null, same: boolean) => void): Promise<boolean>;

  /**
   * Compare data with hash sync
   * @param data The data to be compared
   * @param encrypted The data to be compared to
   */
  export function compareSync(data: string, encrypted: string): boolean;
} 