// This script tests direct database connectivity with environment variables
require('dotenv').config();

const { PrismaClient } = require('@prisma/client');

console.log('Environment variables:');
console.log('NODE_ENV:', process.env.NODE_ENV);
console.log('DATABASE_URL:', process.env.DATABASE_URL ? process.env.DATABASE_URL.replace(/\/\/(.+?):.+?@/, '//***:***@') : 'Not set');
console.log('NEXTAUTH_URL:', process.env.NEXTAUTH_URL || 'Not set');

async function testDatabaseConnection() {
  console.log('\nTesting database connection directly...');
  
  const prisma = new PrismaClient({
    log: ['query', 'info', 'warn', 'error'],
  });
  
  try {
    console.log('Attempting to connect to the database...');
    
    // Simple query to test connectivity
    const hospitalCount = await prisma.hospital.count();
    console.log(`Connection successful! Found ${hospitalCount} hospitals.`);
    
    // Get user count
    const userCount = await prisma.user.count();
    console.log(`Found ${userCount} users.`);
    
    // List super admin users
    const superAdmins = await prisma.user.findMany({
      where: { role: 'super_admin' },
      select: {
        id: true,
        username: true,
        email: true,
        role: true
      }
    });
    console.log('Super admin users:', superAdmins);
    
    return { success: true, hospitalCount, userCount, superAdmins };
  } catch (error) {
    console.error('Database connection failed:', error);
    return { 
      success: false, 
      error: error.message,
      code: error.code,
      meta: error.meta
    };
  } finally {
    await prisma.$disconnect();
  }
}

// Run the test
testDatabaseConnection()
  .then(result => {
    console.log('\nTest completed:', result);
    if (!result.success) {
      process.exit(1);
    }
  })
  .catch(err => {
    console.error('Unexpected error:', err);
    process.exit(1);
  }); 