// Check if environment variables are properly loaded
const fs = require('fs');
const path = require('path');
const dotenv = require('dotenv');

// Try to load environment variables
function loadEnv() {
  console.log('Checking environment variables...');
  
  // Check for .env file
  const envPath = path.join(process.cwd(), '.env');
  if (fs.existsSync(envPath)) {
    console.log(`✅ .env file found at ${envPath}`);
    const envContent = fs.readFileSync(envPath, 'utf8');
    console.log('\nContent of .env file:');
    console.log(envContent);
    
    // Parse environment variables
    const envConfig = dotenv.parse(envContent);
    console.log('\nParsed environment variables:');
    console.log(envConfig);
    
    // Load environment variables
    Object.keys(envConfig).forEach(key => {
      process.env[key] = envConfig[key];
    });
  } else {
    console.log(`❌ .env file not found at ${envPath}`);
  }
  
  // Check for .env.local file
  const envLocalPath = path.join(process.cwd(), '.env.local');
  if (fs.existsSync(envLocalPath)) {
    console.log(`✅ .env.local file found at ${envLocalPath}`);
    const envLocalContent = fs.readFileSync(envLocalPath, 'utf8');
    console.log('\nContent of .env.local file:');
    console.log(envLocalContent);
    
    // Parse environment variables
    const envLocalConfig = dotenv.parse(envLocalContent);
    console.log('\nParsed environment variables from .env.local:');
    console.log(envLocalConfig);
    
    // Load environment variables from .env.local (override .env)
    Object.keys(envLocalConfig).forEach(key => {
      process.env[key] = envLocalConfig[key];
    });
  } else {
    console.log(`❌ .env.local file not found at ${envLocalPath}`);
  }
  
  // Check the environment variables that Next.js/NextAuth needs
  console.log('\nChecking critical environment variables:');
  
  const criticalVars = [
    'DATABASE_URL',
    'NEXTAUTH_SECRET',
    'NEXTAUTH_URL'
  ];
  
  let allCriticalVarsFound = true;
  
  criticalVars.forEach(varName => {
    if (process.env[varName]) {
      console.log(`✅ ${varName}: ${process.env[varName]}`);
    } else {
      console.log(`❌ ${varName}: Not set`);
      allCriticalVarsFound = false;
    }
  });
  
  return allCriticalVarsFound;
}

// Main function
function main() {
  console.log('Environment Variables Check');
  console.log('==========================');
  
  const envVarsLoaded = loadEnv();
  
  console.log('\nSummary:');
  if (envVarsLoaded) {
    console.log('✅ All critical environment variables are set.');
  } else {
    console.log('❌ Some critical environment variables are missing.');
    console.log('Please make sure your .env file contains the following variables:');
    console.log('- DATABASE_URL: MySQL connection string');
    console.log('- NEXTAUTH_SECRET: Secret for NextAuth.js');
    console.log('- NEXTAUTH_URL: URL of your application (e.g., http://localhost:3000)');
  }
}

main(); 