import { PrismaClient } from '@prisma/client';

const prisma = new PrismaClient();

async function main() {
  try {
    // Get hospital IDs
    const hospitals = await prisma.hospital.findMany({
      select: { id: true },
    });

    if (hospitals.length === 0) {
      console.log('No hospitals found in the database. Please run the main seed script first.');
      return;
    }

    const hospital1Id = hospitals[0].id;
    const hospital2Id = hospitals.length > 1 ? hospitals[1].id : hospital1Id;

    // Clear existing data
    await prisma.regulatoryStandard.deleteMany();
    await prisma.regulatoryNews.deleteMany();

    // Seed regulatory standards for hospital 1
    const standards1 = [
      { 
        name: "ISO 9001:2015", 
        category: "Quality Management",
        relevance: "high",
        lastUpdated: new Date('2023-01-15'),
        description: "Specifies requirements for a quality management system. Helps organizations ensure they meet customer and regulatory requirements.",
        hospital_id: hospital1Id
      },
      { 
        name: "HIPAA", 
        category: "Data Privacy",
        relevance: "high",
        lastUpdated: new Date('2023-03-22'),
        description: "Health Insurance Portability and Accountability Act - Sets standards for protecting sensitive patient data.",
        hospital_id: hospital1Id
      },
      { 
        name: "GDPR", 
        category: "Data Protection",
        relevance: "medium",
        lastUpdated: new Date('2023-04-10'),
        description: "General Data Protection Regulation - EU regulation on data protection and privacy.",
        hospital_id: hospital1Id
      },
      { 
        name: "JCI 7th Edition", 
        category: "Healthcare Quality",
        relevance: "high",
        lastUpdated: new Date('2023-05-05'),
        description: "Joint Commission International standards for hospital accreditation.",
        hospital_id: hospital1Id
      },
      { 
        name: "FDA 21 CFR Part 11", 
        category: "Electronic Records",
        relevance: "medium",
        lastUpdated: new Date('2023-02-18'),
        description: "Regulations on electronic records and electronic signatures.",
        hospital_id: hospital1Id
      },
      { 
        name: "OSHA", 
        category: "Workplace Safety",
        relevance: "low",
        lastUpdated: new Date('2023-03-10'),
        description: "Occupational Safety and Health Administration standards for workplace safety.",
        hospital_id: hospital1Id
      }
    ];

    // Seed regulatory standards for hospital 2
    const standards2 = [
      { 
        name: "ISO 9001:2015", 
        category: "Quality Management",
        relevance: "medium",
        lastUpdated: new Date('2023-02-10'),
        description: "Specifies requirements for a quality management system. Helps organizations ensure they meet customer and regulatory requirements.",
        hospital_id: hospital2Id
      },
      { 
        name: "HIPAA", 
        category: "Data Privacy",
        relevance: "high",
        lastUpdated: new Date('2023-04-15'),
        description: "Health Insurance Portability and Accountability Act - Sets standards for protecting sensitive patient data.",
        hospital_id: hospital2Id
      },
      { 
        name: "JCI 7th Edition", 
        category: "Healthcare Quality",
        relevance: "high",
        lastUpdated: new Date('2023-05-20'),
        description: "Joint Commission International standards for hospital accreditation.",
        hospital_id: hospital2Id
      }
    ];

    // Combine and insert all standards
    const allStandards = [...standards1, ...standards2];
    
    for (const standard of allStandards) {
      await prisma.regulatoryStandard.create({
        data: standard,
      });
    }

    console.log(`Created ${allStandards.length} regulatory standards`);

    // Seed regulatory news
    const newsItems = [
      {
        title: 'JCI announces 7th Edition Standard updates',
        source: 'Joint Commission International',
        date: new Date('2023-05-12'),
        category: 'Standards Update',
        content: 'Joint Commission International has announced significant updates to their 7th Edition Standards, focusing on patient safety and quality improvement.',
        url: 'https://www.jointcommissioninternational.org',
        isNew: true,
        hospital_id: null // Visible to all hospitals
      },
      {
        title: 'FDA releases new guidance on medical device reporting',
        source: 'U.S. FDA',
        date: new Date('2023-05-10'),
        category: 'Regulatory Guidance',
        content: 'The FDA has published new guidance on medical device reporting requirements, with updated timelines and processes.',
        url: 'https://www.fda.gov/medical-devices',
        isNew: false,
        hospital_id: null // Visible to all hospitals
      },
      {
        title: 'GDPR enforcement actions increasing in healthcare sector',
        source: 'EU Commission',
        date: new Date('2023-05-07'),
        category: 'Enforcement',
        content: 'The EU Commission has reported an increase in GDPR enforcement actions specifically targeting healthcare organizations.',
        url: 'https://ec.europa.eu/info/law/law-topic/data-protection_en',
        isNew: false,
        hospital_id: null // Visible to all hospitals
      },
      {
        title: 'WHO publishes updated patient safety framework',
        source: 'World Health Organization',
        date: new Date('2023-05-05'),
        category: 'Standards Update',
        content: 'The World Health Organization has released an updated framework for patient safety, emphasizing a systems approach to reducing errors.',
        url: 'https://www.who.int/health-topics/patient-safety',
        isNew: true,
        hospital_id: null // Visible to all hospitals
      },
      {
        title: 'New patient privacy requirements announced',
        source: 'Regional Health Authority',
        date: new Date('2023-05-09'),
        category: 'Privacy',
        content: 'The Regional Health Authority has announced new patient privacy requirements that will come into effect next quarter.',
        url: null,
        isNew: true,
        hospital_id: hospital1Id // Hospital-specific news
      }
    ];

    for (const news of newsItems) {
      await prisma.regulatoryNews.create({
        data: news,
      });
    }

    console.log(`Created ${newsItems.length} regulatory news items`);

  } catch (error) {
    console.error('Error seeding regulatory data:', error);
    throw error;
  }
}

main()
  .then(async () => {
    await prisma.$disconnect();
  })
  .catch(async (e) => {
    console.error(e);
    await prisma.$disconnect();
    process.exit(1);
  }); 