import { PrismaClient, Prisma } from '@prisma/client';

const prisma = new PrismaClient();

// Define enum values
type SeverityType = 'critical' | 'high' | 'medium' | 'low';
type StatusType = 'open' | 'in_progress' | 'resolved';

async function main() {
  try {
    // Get all hospitals
    const hospitals = await prisma.hospital.findMany();
    
    if (hospitals.length === 0) {
      console.error('No hospitals found in database');
      return;
    }
    
    // Create findings data with 2 of each severity level
    const findingItems = [
      // Critical findings
      { 
        title: 'Unauthorized access to patient records detected',
        department: 'IT Security',
        severity: 'critical' as SeverityType,
        status: 'open' as StatusType,
        description: 'System logs indicate multiple instances of unauthorized access to patient records from external IP addresses.',
        date_reported: new Date('2023-06-15'),
        hospital_id: hospitals[0].id 
      },
      { 
        title: 'Fire suppression system failure in server room',
        department: 'Facilities',
        severity: 'critical' as SeverityType,
        status: 'in_progress' as StatusType,
        description: 'The automated fire suppression system in the main server room failed multiple safety tests.',
        date_reported: new Date('2023-06-10'),
        hospital_id: hospitals.length > 1 ? hospitals[1].id : hospitals[0].id
      },
      
      // High severity findings
      { 
        title: 'Medication administration errors in Pediatrics',
        department: 'Nursing',
        severity: 'high' as SeverityType,
        status: 'open' as StatusType,
        description: 'Multiple incidents of incorrect medication dosing were reported in the pediatric department over the past month.',
        date_reported: new Date('2023-06-08'),
        hospital_id: hospitals[0].id 
      },
      { 
        title: 'Outdated patient consent forms in use',
        department: 'Legal',
        severity: 'high' as SeverityType,
        status: 'in_progress' as StatusType,
        description: 'Several departments are still using outdated patient consent forms that do not comply with current regulations.',
        date_reported: new Date('2023-06-05'),
        hospital_id: hospitals.length > 1 ? hospitals[1].id : hospitals[0].id
      },
      
      // Medium severity findings
      { 
        title: 'Inconsistent patient ID verification process',
        department: 'Administration',
        severity: 'medium' as SeverityType,
        status: 'open' as StatusType,
        description: 'Staff members are not consistently following the required two-step patient identification process.',
        date_reported: new Date('2023-06-03'),
        hospital_id: hospitals[0].id 
      },
      { 
        title: 'Incomplete preventive maintenance records',
        department: 'Laboratory',
        severity: 'medium' as SeverityType,
        status: 'resolved' as StatusType,
        description: 'Preventive maintenance records for laboratory equipment are incomplete or missing for Q1 2023.',
        date_reported: new Date('2023-05-28'),
        hospital_id: hospitals.length > 1 ? hospitals[1].id : hospitals[0].id
      }
    ];
    
    // Add the findings
    for (const item of findingItems) {
      await prisma.auditFinding.create({ data: item });
      console.log(`Created finding: ${item.title}`);
    }
    
    console.log(`Successfully added ${findingItems.length} new findings!`);
  } catch (error) {
    console.error('Error seeding findings data:', error);
    throw error;
  } finally {
    await prisma.$disconnect();
  }
}

main()
  .catch(async (e) => {
    console.error(e);
    await prisma.$disconnect();
    process.exit(1);
  }); 