import { PrismaClient } from '@prisma/client'

const prisma = new PrismaClient()

async function main() {
  try {
    console.log('🌱 Seeding finding options...')

    // Seed Finding Severities
    const severities = [
      {
        value: 'critical',
        label: 'Critical',
        description: 'Critical issues requiring immediate attention',
        color_class: 'bg-red-700 text-white',
        sort_order: 1
      },
      {
        value: 'high',
        label: 'High',
        description: 'High priority issues requiring prompt attention',
        color_class: 'bg-red-500 text-white',
        sort_order: 2
      },
      {
        value: 'medium',
        label: 'Medium',
        description: 'Medium priority issues requiring attention',
        color_class: 'bg-amber-500 text-white',
        sort_order: 3
      },
      {
        value: 'low',
        label: 'Low',
        description: 'Low priority issues for future consideration',
        color_class: 'bg-blue-500 text-white',
        sort_order: 4
      }
    ]

    for (const severity of severities) {
      await prisma.findingSeverity.upsert({
        where: { value: severity.value },
        update: severity,
        create: severity
      })
      console.log(`✅ Created/updated severity: ${severity.label}`)
    }

    // Seed Finding Statuses
    const statuses = [
      {
        value: 'open',
        label: 'Open',
        description: 'Finding is open and requires action',
        color_class: 'text-gray-600 border-gray-300',
        sort_order: 1
      },
      {
        value: 'in_progress',
        label: 'In Progress',
        description: 'Finding is being actively addressed',
        color_class: 'text-blue-600 border-blue-300',
        sort_order: 2
      },
      {
        value: 'resolved',
        label: 'Resolved',
        description: 'Finding has been resolved',
        color_class: 'text-green-600 border-green-300',
        sort_order: 3
      }
    ]

    for (const status of statuses) {
      await prisma.findingStatus.upsert({
        where: { value: status.value },
        update: status,
        create: status
      })
      console.log(`✅ Created/updated status: ${status.label}`)
    }

    console.log('🎉 Finding options seeding completed!')

  } catch (error) {
    console.error('❌ Error seeding finding options:', error)
    throw error
  } finally {
    await prisma.$disconnect()
  }
}

main()
  .catch((e) => {
    console.error(e)
    process.exit(1)
  }) 