import { PrismaClient } from '@prisma/client';

const prisma = new PrismaClient();

async function main() {
  try {
    console.log('Seeding document categories and documents...');
    
    // Get existing hospitals
    const hospitals = await prisma.hospital.findMany();
    
    if (hospitals.length === 0) {
      console.error('No hospitals found. Please run the main seed file first.');
      return;
    }
    
    // Seed document categories for each hospital
    for (const hospital of hospitals) {
      console.log(`Adding document categories for hospital: ${hospital.name}`);
      
      const categories = [
        {
          name: 'Policies',
          icon: 'FileText',
          description: 'Organizational policies for compliance and governance',
          hospital_id: hospital.id
        },
        {
          name: 'Procedures',
          icon: 'FileCheck',
          description: 'Step-by-step procedures for clinical operations',
          hospital_id: hospital.id
        },
        {
          name: 'Forms',
          icon: 'FileText',
          description: 'Standard forms for patient care and administration',
          hospital_id: hospital.id
        },
        {
          name: 'Training Materials',
          icon: 'GraduationCap',
          description: 'Staff training resources and compliance education',
          hospital_id: hospital.id
        },
        {
          name: 'Certifications',
          icon: 'Award',
          description: 'Certifications and accreditations',
          hospital_id: hospital.id
        },
        {
          name: 'Reports',
          icon: 'BarChart',
          description: 'Audit reports and compliance assessments',
          hospital_id: hospital.id
        }
      ];
      
      // Create categories
      for (const category of categories) {
        const existingCategory = await prisma.documentCategory.findFirst({
          where: {
            name: category.name,
            hospital_id: hospital.id
          }
        });
        
        if (!existingCategory) {
          await prisma.documentCategory.create({
            data: category
          });
          console.log(`Created category: ${category.name}`);
        } else {
          console.log(`Category already exists: ${category.name}`);
        }
      }
      
      // Get created categories
      const createdCategories = await prisma.documentCategory.findMany({
        where: {
          hospital_id: hospital.id
        }
      });
      
      console.log(`Found ${createdCategories.length} document categories for ${hospital.name}`);
      
      // Seed sample documents
      if (createdCategories.length > 0) {
        const policyCategory = createdCategories.find(c => c.name === 'Policies');
        const procedureCategory = createdCategories.find(c => c.name === 'Procedures');
        const formCategory = createdCategories.find(c => c.name === 'Forms');
        
        if (policyCategory && procedureCategory && formCategory) {
          const documents = [
            {
              title: 'HIPAA Compliance Manual v3.2',
              fileName: 'hipaa_compliance_manual.pdf',
              status: 'active',
              department: 'Clinical Operations',
              updatedBy: 'admin@example.com',
              category_id: policyCategory.id,
              hospital_id: hospital.id
            },
            {
              title: 'Infection Control Protocol 2023',
              fileName: 'infection_control_2023.pdf',
              status: 'active',
              department: 'Clinical Operations',
              updatedBy: 'admin@example.com',
              category_id: procedureCategory.id,
              hospital_id: hospital.id
            },
            {
              title: 'Patient Data Access Request Form',
              fileName: 'patient_data_access.pdf',
              status: 'active',
              department: 'Legal',
              updatedBy: 'admin@example.com',
              category_id: formCategory.id,
              hospital_id: hospital.id
            },
            {
              title: 'Medical Waste Disposal Procedure',
              fileName: 'waste_disposal.pdf',
              status: 'review',
              department: 'Environmental Services',
              updatedBy: 'admin@example.com',
              category_id: procedureCategory.id,
              hospital_id: hospital.id
            },
            {
              title: 'Patient Privacy Policy',
              fileName: 'patient_privacy_policy.pdf',
              status: 'active',
              department: 'Legal',
              updatedBy: 'admin@example.com',
              category_id: policyCategory.id,
              hospital_id: hospital.id
            }
          ];
          
          // Create documents
          for (const doc of documents) {
            const existingDoc = await prisma.document.findFirst({
              where: {
                title: doc.title,
                hospital_id: hospital.id
              }
            });
            
            if (!existingDoc) {
              await prisma.document.create({
                data: doc
              });
              console.log(`Created document: ${doc.title}`);
            } else {
              console.log(`Document already exists: ${doc.title}`);
            }
          }
          
          const docCount = await prisma.document.count({
            where: {
              hospital_id: hospital.id
            }
          });
          console.log(`Found ${docCount} documents for ${hospital.name}`);
        }
      }
    }
    
    console.log('Document categories and documents seeded successfully!');
  } catch (error) {
    console.error('Error seeding document categories and documents:', error);
    throw error;
  }
}

main()
  .then(async () => {
    await prisma.$disconnect();
  })
  .catch(async (e) => {
    console.error(e);
    await prisma.$disconnect();
    process.exit(1);
  }); 