import { PrismaClient } from '@prisma/client'

const prisma = new PrismaClient()

// Default departments to seed
const departments = [
  { name: 'Nursing', description: 'Nursing and patient care staff' },
  { name: 'Pharmacy', description: 'Medication management and pharmacy services' },
  { name: 'Laboratory', description: 'Medical testing and laboratory services' },
  { name: 'Radiology', description: 'Imaging and diagnostic services' },
  { name: 'Emergency', description: 'Emergency medicine and urgent care' },
  { name: 'Surgery', description: 'Surgical services and operating rooms' },
  { name: 'Facilities', description: 'Building maintenance and operations' },
  { name: 'Information Technology', description: 'IT systems and technical support' },
  { name: 'Administration', description: 'Hospital administration and management' },
  { name: 'Human Resources', description: 'Staffing and personnel management' },
  { name: 'Quality Management', description: 'Quality assurance and improvement' },
  { name: 'Infection Control', description: 'Infection prevention and control measures' }
]

async function main() {
  console.log('Seeding departments...')
  
  // Get all hospitals
  const hospitals = await prisma.hospital.findMany()
  
  if (hospitals.length === 0) {
    console.log('No hospitals found. Please seed hospitals first.')
    return
  }
  
  // For each hospital, create departments
  for (const hospital of hospitals) {
    console.log(`Adding departments for hospital: ${hospital.name}`)
    
    // Check if hospital already has departments
    const existingDeptCount = await prisma.department.count({
      where: {
        hospital_id: hospital.id
      }
    })
    
    if (existingDeptCount > 0) {
      console.log(`Hospital ${hospital.name} already has ${existingDeptCount} departments. Skipping.`)
      continue
    }
    
    // Create departments for this hospital
    for (const dept of departments) {
      await prisma.department.create({
        data: {
          name: dept.name,
          description: dept.description,
          hospital_id: hospital.id
        }
      })
    }
    
    console.log(`Created ${departments.length} departments for hospital: ${hospital.name}`)
  }
  
  console.log('Departments seeding complete!')
}

main()
  .catch((e) => {
    console.error(e)
    process.exit(1)
  })
  .finally(async () => {
    await prisma.$disconnect()
  }) 