'use client'

import React, { createContext, useContext, useState, ReactNode, useEffect } from 'react';
import { useSession } from 'next-auth/react';

export type UserRole = 'super_admin' | 'hospital' | 'user';

export interface UserPermission {
  hospitalId: string;
  accessLevel: 'full' | 'read' | 'none';
}

export interface User {
  id: string;
  name?: string;
  email: string;
  username: string;
  role: UserRole;
  permissions: UserPermission[];
  hospital_id?: string;
}

interface UserContextType {
  currentUser: User | null;
  users: User[];
  isSuperAdmin: boolean;
  isHospitalAdmin: boolean;
  isLoggedIn: boolean;
  hasAccessToHospital: (hospitalId: string) => boolean;
  getAccessibleHospitals: () => string[];
  updateUserPermissions: (userId: string, permissions: UserPermission[]) => void;
  loading: boolean;
  error: string | null;
  refetchUser: () => Promise<void>;
}

const UserContext = createContext<UserContextType | undefined>(undefined);

export const UserProvider = ({ children }: { children: ReactNode }) => {
  const { data: session, status } = useSession();
  const [currentUser, setCurrentUser] = useState<User | null>(null);
  const [users, setUsers] = useState<User[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  // Fetch current user data from the API
  const fetchCurrentUser = async () => {
    if (!session?.user?.email) {
      setCurrentUser(null);
      setLoading(false);
      return;
    }

    setLoading(true);
    setError(null);

    try {
      const response = await fetch('/api/auth/check');
      
      if (!response.ok) {
        throw new Error('Failed to fetch user data');
      }
      
      const userData = await response.json();
      
      // Transform the user data to match our interface
      const user: User = {
        id: userData.id.toString(),
        name: userData.name || session.user.name || '',
        email: userData.email,
        username: userData.username,
        role: userData.role as UserRole,
        permissions: userData.hospital_id ? [
          { hospitalId: userData.hospital_id.toString(), accessLevel: 'full' }
        ] : [],
        hospital_id: userData.hospital_id?.toString()
      };
      
      setCurrentUser(user);
    } catch (err) {
      console.error('Error fetching user data:', err);
      setError(err instanceof Error ? err.message : 'Failed to fetch user data');
      setCurrentUser(null);
    } finally {
      setLoading(false);
    }
  };

  // Fetch users (for admin functionality)
  const fetchUsers = async () => {
    if (!session?.user?.email || currentUser?.role !== 'super_admin') {
      return;
    }

    try {
      const response = await fetch('/api/admin/users');
      
      if (!response.ok) {
        throw new Error('Failed to fetch users');
      }
      
      const usersData = await response.json();
      setUsers(usersData);
    } catch (err) {
      console.error('Error fetching users:', err);
      // Don't set error for users fetch as it's not critical for the app
    }
  };

  // Fetch user data when session changes
  useEffect(() => {
    if (status === 'loading') {
      setLoading(true);
      return;
    }
    
    if (status === 'authenticated') {
      fetchCurrentUser();
    } else {
      setCurrentUser(null);
      setUsers([]);
      setLoading(false);
    }
  }, [session, status]);

  // Fetch users when current user is loaded and is super admin
  useEffect(() => {
    if (currentUser?.role === 'super_admin') {
      fetchUsers();
    }
  }, [currentUser]);

  const isLoggedIn = status === 'authenticated' && !!currentUser;
  const isSuperAdmin = currentUser?.role === 'super_admin';
  const isHospitalAdmin = currentUser?.role === 'hospital';

  const hasAccessToHospital = (hospitalId: string): boolean => {
    if (!currentUser) return false;
    if (isSuperAdmin) return true;
    
    // Check if user has direct access to this hospital
    if (currentUser.hospital_id === hospitalId) return true;
    
    const permission = currentUser.permissions.find(p => p.hospitalId === hospitalId);
    return !!permission && permission.accessLevel !== 'none';
  };

  const getAccessibleHospitals = (): string[] => {
    if (!currentUser) return [];
    if (isSuperAdmin) return []; // Superadmin has access to all hospitals
    
    // If user has a direct hospital assignment, return that
    if (currentUser.hospital_id) {
      return [currentUser.hospital_id];
    }
    
    return currentUser.permissions
      .filter(p => p.accessLevel !== 'none')
      .map(p => p.hospitalId);
  };

  const updateUserPermissions = (userId: string, permissions: UserPermission[]) => {
    // In a real application, this would make an API call to update the database
    console.log(`Updating permissions for user ${userId}:`, permissions);
  };
  
  const refetchUser = async () => {
    await fetchCurrentUser();
  };

  return (
    <UserContext.Provider
      value={{
        currentUser,
        users,
        isSuperAdmin,
        isHospitalAdmin,
        isLoggedIn,
        hasAccessToHospital,
        getAccessibleHospitals,
        updateUserPermissions,
        loading,
        error,
        refetchUser
      }}
    >
      {children}
    </UserContext.Provider>
  );
};

export const useUser = () => {
  const context = useContext(UserContext);
  if (context === undefined) {
    throw new Error('useUser must be used within a UserProvider');
  }
  return context;
}; 