import { useState, useEffect } from 'react';
import { toast } from 'sonner';

export interface TrainingEvent {
  id: string;
  title: string;
  date: string;
  instructor: string;
  location: string;
  department: string;
  duration: string;
  capacity: number;
  enrolled: number;
  required: boolean;
}

export interface TrainingModule {
  id: string;
  title: string;
  completionRate: number;
  dueDate: string;
  department: string;
  priority: 'high' | 'medium' | 'low';
}

interface TrainingDataState {
  trainingEvents: TrainingEvent[];
  trainingModules: TrainingModule[];
  staffCompliance: {
    overallRate: number;
    departments: {
      name: string;
      complianceRate: number;
      staffCount: number;
    }[];
  };
  loading: {
    events: boolean;
    modules: boolean;
    compliance: boolean;
  };
  error: {
    events: string | null;
    modules: string | null;
    compliance: string | null;
  };
}

const initialState: TrainingDataState = {
  trainingEvents: [],
  trainingModules: [],
  staffCompliance: {
    overallRate: 0,
    departments: [],
  },
  loading: {
    events: false,
    modules: false,
    compliance: false,
  },
  error: {
    events: null,
    modules: null,
    compliance: null,
  },
};

export const useTrainingData = () => {
  const [state, setState] = useState<TrainingDataState>(initialState);

  // Fetch all training data
  useEffect(() => {
    fetchTrainingEvents();
    fetchStaffCompliance();
    // In a real app, you would also fetch other data
    // fetchTrainingModules();
    
    // For now, generate mock data for modules
    generateMockTrainingModules();
  }, []);

  // Fetch training events from the API
  const fetchTrainingEvents = async () => {
    console.log('=== fetchTrainingEvents - Start ===');
    setState(prev => ({ ...prev, loading: { ...prev.loading, events: true } }));
    
    try {
      console.log('Fetching training events from API...');
      const response = await fetch('/api/schedule/training');
      console.log('Training events API response status:', response.status);
      console.log('Training events API response headers:', Object.fromEntries(response.headers.entries()));
      
      if (!response.ok) {
        const errorText = await response.text();
        console.error('Training events API error response:', errorText);
        throw new Error(`Failed to fetch training events: ${response.status} ${response.statusText}`);
      }
      
      const events = await response.json();
      console.log('Training events fetched successfully:', events?.length || 0, 'events');
      console.log('Sample event:', events?.[0]);
      
      // Format dates for display
      const formattedEvents = events?.map((item: any) => ({
        ...item,
        date: item.date ? new Date(item.date).toISOString().split('T')[0] : '',
      })) || [];
      
      setState(prev => ({
        ...prev,
        trainingEvents: formattedEvents,
        loading: { ...prev.loading, events: false },
        error: { ...prev.error, events: null }
      }));
      
      console.log('=== fetchTrainingEvents - Success ===');
    } catch (error) {
      console.error('=== fetchTrainingEvents - Error ===');
      console.error('Error fetching training events:', error);
      setState(prev => ({
        ...prev,
        trainingEvents: [],
        loading: { ...prev.loading, events: false },
        error: { ...prev.error, events: error instanceof Error ? error.message : 'Failed to fetch training events' }
      }));
    }
  };

  // Generate mock training modules data for demonstration
  const generateMockTrainingModules = () => {
    setState(prev => ({
      ...prev,
      loading: { ...prev.loading, modules: true },
    }));

    // Simulate API delay
    setTimeout(() => {
      const modules: TrainingModule[] = [
        {
          id: "tm1",
          title: "HIPAA Compliance Basics",
          completionRate: 78,
          dueDate: "2023-06-30",
          department: "All Staff",
          priority: "high"
        },
        {
          id: "tm2",
          title: "Infection Control Procedures",
          completionRate: 92,
          dueDate: "2023-06-15",
          department: "Clinical Staff",
          priority: "high"
        },
        {
          id: "tm3",
          title: "Data Security Best Practices",
          completionRate: 65,
          dueDate: "2023-07-10",
          department: "All Staff",
          priority: "medium"
        },
        {
          id: "tm4",
          title: "Patient Rights and Privacy",
          completionRate: 81,
          dueDate: "2023-06-25",
          department: "Patient-Facing Staff",
          priority: "high"
        },
        {
          id: "tm5",
          title: "Emergency Response Training",
          completionRate: 74,
          dueDate: "2023-07-15",
          department: "All Staff",
          priority: "high"
        }
      ];

      setState(prev => ({
        ...prev,
        trainingModules: modules,
        loading: { ...prev.loading, modules: false },
      }));
    }, 500);
  };

  // Generate mock staff compliance data for demonstration
  const generateMockStaffCompliance = () => {
    setState(prev => ({
      ...prev,
      loading: { ...prev.loading, compliance: true },
    }));

    // Simulate API delay
    setTimeout(() => {
      const departments = [
        { name: "Nursing", complianceRate: 88, staffCount: 45 },
        { name: "Physicians", complianceRate: 76, staffCount: 32 },
        { name: "Administration", complianceRate: 94, staffCount: 18 },
        { name: "IT", complianceRate: 82, staffCount: 12 },
        { name: "Facilities", complianceRate: 79, staffCount: 15 },
        { name: "Laboratory", complianceRate: 85, staffCount: 20 }
      ];

      // Calculate overall compliance rate
      const totalStaff = departments.reduce((sum, dept) => sum + dept.staffCount, 0);
      const weightedCompliance = departments.reduce((sum, dept) => 
        sum + (dept.complianceRate * dept.staffCount), 0);
      const overallRate = totalStaff > 0 ? Math.round(weightedCompliance / totalStaff) : 0;

      setState(prev => ({
        ...prev,
        staffCompliance: {
          overallRate,
          departments,
        },
        loading: { ...prev.loading, compliance: false },
      }));
    }, 700);
  };

  // Add new methods for staff compliance management
  const fetchStaffCompliance = async () => {
    setState(prev => ({
      ...prev,
      loading: { ...prev.loading, compliance: true },
      error: { ...prev.error, compliance: null }
    }));
    
    try {
      const response = await fetch('/api/schedule/staff-compliance');
      
      if (!response.ok) {
        let errorMessage = 'Failed to fetch staff compliance data';
        
        // Safely try to parse error response
        try {
          const responseText = await response.text();
          if (responseText) {
            const errorData = JSON.parse(responseText);
            errorMessage = errorData.message || errorMessage;
          }
        } catch (parseError) {
          console.error('Error parsing error response:', parseError);
        }
        
        throw new Error(errorMessage);
      }
      
      const data = await response.json();
      
      // The API returns data in the format we need
      setState(prev => ({
        ...prev,
        staffCompliance: data,
        loading: { ...prev.loading, compliance: false }
      }));

      return data;
    } catch (error) {
      console.error('Error fetching staff compliance:', error);
      
      setState(prev => ({
        ...prev,
        loading: { ...prev.loading, compliance: false },
        error: { 
          ...prev.error, 
          compliance: error instanceof Error ? error.message : 'An error occurred' 
        }
      }));
      
      // Only generate mock data if we get a 404 (no data yet)
      if (error instanceof Error && error.message.includes('404')) {
        generateMockStaffCompliance();
      } else {
        toast.error('Failed to load staff compliance data');
      }
      
      return null;
    }
  };

  // Create a new staff compliance record
  const createStaffCompliance = async (data: {
    name: string;
    staffCount: number;
    complianceRate: number;
    notes?: string;
  }): Promise<boolean> => {
    try {
      console.log('Creating staff compliance record:', data);
      
      const requestBody = {
        department: data.name,
        staffCount: data.staffCount,
        complianceRate: data.complianceRate,
        notes: data.notes
      };
      
      console.log('Request body:', JSON.stringify(requestBody));
      
      const response = await fetch('/api/schedule/staff-compliance', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(requestBody),
      });

      console.log('Response status:', response.status);
      console.log('Response status text:', response.statusText);
      
      // Store the response body text
      const responseText = await response.text();
      console.log('Response body:', responseText);
      
      // First check if response is ok
      if (!response.ok) {
        let errorMessage = 'Failed to create staff compliance record';
        
        try {
          // Try to parse the response text as JSON
          if (responseText) {
            const errorData = JSON.parse(responseText);
            errorMessage = errorData.message || errorMessage;
          }
        } catch (parseError) {
          console.error('Error parsing error response:', parseError);
        }
        
        throw new Error(errorMessage);
      }

      // Only try to parse JSON if the response is ok and the text is not empty
      const responseData = responseText ? JSON.parse(responseText) : {};
      
      // Refresh staff compliance data after creating
      await fetchStaffCompliance();
      toast.success(responseData.message || 'Staff compliance record created successfully');
      return true;
    } catch (error) {
      console.error('Error creating staff compliance record:', error);
      toast.error(error instanceof Error ? error.message : 'Failed to create staff compliance record');
      return false;
    }
  };

  // Update an existing staff compliance record
  const updateStaffCompliance = async (data: {
    id: number;
    name?: string;
    staffCount?: number;
    complianceRate?: number;
    notes?: string;
  }): Promise<boolean> => {
    try {
      console.log('Updating staff compliance record:', data);
      
      const requestBody = {
        id: data.id,
        department: data.name,
        staffCount: data.staffCount,
        complianceRate: data.complianceRate,
        notes: data.notes
      };
      
      console.log('Request body:', JSON.stringify(requestBody));
      
      const response = await fetch('/api/schedule/staff-compliance', {
        method: 'PATCH',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(requestBody),
      });

      console.log('Response status:', response.status);
      console.log('Response status text:', response.statusText);
      
      // Store the response body text
      const responseText = await response.text();
      console.log('Response body:', responseText);
      
      // First check if response is ok
      if (!response.ok) {
        let errorMessage = 'Failed to update staff compliance record';
        
        try {
          // Try to parse the response text as JSON
          if (responseText) {
            const errorData = JSON.parse(responseText);
            errorMessage = errorData.message || errorMessage;
          }
        } catch (parseError) {
          console.error('Error parsing error response:', parseError);
        }
        
        throw new Error(errorMessage);
      }

      // Only try to parse JSON if the response is ok and the text is not empty
      const responseData = responseText ? JSON.parse(responseText) : {};
      
      // Refresh staff compliance data after updating
      await fetchStaffCompliance();
      toast.success(responseData.message || 'Staff compliance record updated successfully');
      return true;
    } catch (error) {
      console.error('Error updating staff compliance record:', error);
      toast.error(error instanceof Error ? error.message : 'Failed to update staff compliance record');
      return false;
    }
  };

  // Delete a staff compliance record
  const deleteStaffCompliance = async (id: number): Promise<boolean> => {
    try {
      console.log('Deleting staff compliance record:', id);
      const response = await fetch(`/api/schedule/staff-compliance?id=${id}`, {
        method: 'DELETE',
      });

      console.log('Response status:', response.status);
      console.log('Response status text:', response.statusText);
      
      // Store the response body text
      const responseText = await response.text();
      console.log('Response body:', responseText);
      
      // First check if response is ok
      if (!response.ok) {
        let errorMessage = 'Failed to delete staff compliance record';
        
        try {
          // Try to parse the response text as JSON
          if (responseText) {
            const errorData = JSON.parse(responseText);
            errorMessage = errorData.message || errorMessage;
          }
        } catch (parseError) {
          console.error('Error parsing error response:', parseError);
        }
        
        throw new Error(errorMessage);
      }

      // Only try to parse JSON if the response is ok and the text is not empty
      const responseData = responseText ? JSON.parse(responseText) : {};
      
      // Refresh staff compliance data after deleting
      await fetchStaffCompliance();
      toast.success(responseData.message || 'Staff compliance record deleted successfully');
      return true;
    } catch (error) {
      console.error('Error deleting staff compliance record:', error);
      toast.error(error instanceof Error ? error.message : 'Failed to delete staff compliance record');
      return false;
    }
  };

  // Create a new training event
  const createTrainingEvent = async (training: Omit<TrainingEvent, 'id'>): Promise<boolean> => {
    try {
      console.log('Creating training event with data:', training);
      
      const response = await fetch('/api/schedule/training', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(training),
      });

      if (!response.ok) {
        let errorMessage = 'Failed to create training event';
        
        // Safely try to parse error response
        try {
          const responseText = await response.text();
          console.log('Training API error response:', responseText);
          
          if (responseText) {
            const errorData = JSON.parse(responseText);
            // Use the detailed error message if available
            errorMessage = errorData.details || errorData.error || errorMessage;
          }
        } catch (parseError) {
          console.error('Error parsing error response:', parseError);
        }
        
        throw new Error(errorMessage);
      }

      // Refresh training events after creating
      await fetchTrainingEvents();
      toast.success('Training event created successfully');
      return true;
    } catch (error) {
      console.error('Error creating training event:', error);
      toast.error(error instanceof Error ? error.message : 'Failed to create training event');
      return false;
    }
  };

  // Update an existing training event
  const updateTrainingEvent = async (training: TrainingEvent): Promise<boolean> => {
    try {
      const response = await fetch('/api/schedule/training', {
        method: 'PATCH',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(training),
      });

      if (!response.ok) {
        let errorMessage = 'Failed to update training event';
        
        // Safely try to parse error response
        try {
          const responseText = await response.text();
          if (responseText) {
            const errorData = JSON.parse(responseText);
            errorMessage = errorData.message || errorMessage;
          }
        } catch (parseError) {
          console.error('Error parsing error response:', parseError);
        }
        
        throw new Error(errorMessage);
      }

      // Refresh training events after updating
      await fetchTrainingEvents();
      toast.success('Training event updated successfully');
      return true;
    } catch (error) {
      console.error('Error updating training event:', error);
      toast.error(error instanceof Error ? error.message : 'Failed to update training event');
      return false;
    }
  };

  // Delete a training event
  const deleteTrainingEvent = async (id: string): Promise<boolean> => {
    try {
      const response = await fetch(`/api/schedule/training?id=${id}`, {
        method: 'DELETE',
      });

      if (!response.ok) {
        let errorMessage = 'Failed to delete training event';
        
        // Safely try to parse error response
        try {
          const responseText = await response.text();
          if (responseText) {
            const errorData = JSON.parse(responseText);
            errorMessage = errorData.message || errorMessage;
          }
        } catch (parseError) {
          console.error('Error parsing error response:', parseError);
        }
        
        throw new Error(errorMessage);
      }

      // Refresh training events after deleting
      await fetchTrainingEvents();
      toast.success('Training event deleted successfully');
      return true;
    } catch (error) {
      console.error('Error deleting training event:', error);
      toast.error(error instanceof Error ? error.message : 'Failed to delete training event');
      return false;
    }
  };

  // Calculate training metrics
  const getTrainingMetrics = () => {
    const now = new Date();
    
    // Count upcoming events in the next 30 days
    const upcomingEvents = state.trainingEvents.filter(event => {
      const eventDate = new Date(event.date);
      const diffTime = eventDate.getTime() - now.getTime();
      const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));
      return diffDays >= 0 && diffDays <= 30;
    }).length;
    
    // Calculate average completion rate of training modules
    const avgCompletionRate = state.trainingModules.length > 0 
      ? Math.round(state.trainingModules.reduce((sum, module) => sum + module.completionRate, 0) / state.trainingModules.length) 
      : 0;
    
    // Count high priority training modules
    const highPriorityModules = state.trainingModules.filter(
      module => module.priority === 'high'
    ).length;
    
    return {
      upcomingEvents,
      avgCompletionRate,
      highPriorityModules,
      totalEvents: state.trainingEvents.length,
      totalModules: state.trainingModules.length,
    };
  };

  return {
    ...state,
    // CRUD operations
    createTrainingEvent,
    updateTrainingEvent,
    deleteTrainingEvent,
    // Staff compliance operations
    createStaffCompliance,
    updateStaffCompliance,
    deleteStaffCompliance,
    // Refresh methods
    refreshTrainingEvents: fetchTrainingEvents,
    refreshStaffCompliance: fetchStaffCompliance,
    // Metrics
    getTrainingMetrics,
  };
}; 