'use client'

import { useHospital } from '@/lib/context/HospitalContext';
import { useUser } from '@/lib/context/UserContext';
import { useCallback, useEffect, useState } from 'react';

/**
 * Custom hook to handle hospital selection consistently across components
 * 
 * Use this hook in any component that needs to access or filter data based on 
 * the selected hospital, particularly for super admins who can switch hospitals.
 */
export function useSelectedHospital() {
  const { currentHospital, loading, error, refetchHospitals } = useHospital();
  const { isSuperAdmin } = useUser();
  const [previousHospitalId, setPreviousHospitalId] = useState<string | null>(null);

  // Track hospital changes to allow components to refresh data
  useEffect(() => {
    if (currentHospital) {
      const currentId = String(currentHospital.id);
      
      // If the hospital changed, update the previous ID
      if (previousHospitalId !== null && previousHospitalId !== currentId) {
        console.log(`Hospital changed from ${previousHospitalId} to ${currentId}`);
      }
      
      setPreviousHospitalId(currentId);
    }
  }, [currentHospital, previousHospitalId]);

  /**
   * Get the current hospital ID for API calls or data filtering
   */
  const getSelectedHospitalId = useCallback(() => {
    if (!isSuperAdmin) return undefined; // Not applicable for non-super admins
    if (!currentHospital) return null; // No hospital selected
    return currentHospital.id;
  }, [currentHospital, isSuperAdmin]);

  /**
   * Check if data should be filtered by the current hospital
   * For data filtering in components, use: data.filter(item => filterByHospital(item.hospitalId))
   */
  const filterByHospital = useCallback((itemHospitalId: string) => {
    if (!isSuperAdmin) return true; // Non-super admins see their own hospital data
    if (!currentHospital) return false; // No hospital selected means don't show data
    
    // Compare as strings to avoid type mismatches
    const currentId = String(currentHospital.id);
    const itemId = String(itemHospitalId);
    return currentId === itemId; // Only show data for selected hospital
  }, [currentHospital, isSuperAdmin]);

  /**
   * Get URL parameter to add to API calls
   * For fetch calls, use: fetch(`/api/endpoint${getHospitalUrlParam()}`)
   */
  const getHospitalUrlParam = useCallback(() => {
    if (!isSuperAdmin || !currentHospital) return '';
    
    // Ensure we're using a string representation of the hospital ID
    const hospitalId = String(currentHospital.id);
    const param = `?hospitalId=${encodeURIComponent(hospitalId)}`;
    
    // For debugging
    console.log(`Adding hospital URL param: ${param}`);
    
    return param;
  }, [currentHospital, isSuperAdmin]);

  /**
   * Add hospital ID to fetch options
   * For fetch calls, use: fetch('/api/endpoint', getHospitalFetchOptions({ method: 'POST', body: data }))
   */
  const getHospitalFetchOptions = useCallback((options: RequestInit = {}) => {
    if (!isSuperAdmin || !currentHospital) return options;
    
    // Ensure we're using a string representation of the hospital ID
    const hospitalId = String(currentHospital.id);
    
    // Create headers if they don't exist
    const headers = new Headers(options.headers || {});
    headers.set('X-Hospital-ID', hospitalId);
    
    // For debugging
    console.log(`Adding hospital ID ${hospitalId} to request headers`);
    
    // Add hospital ID to the body for POST/PUT requests
    let modifiedBody = options.body;
    if (options.method && ['POST', 'PUT'].includes(options.method) && typeof options.body === 'string') {
      try {
        const bodyObj = JSON.parse(options.body);
        bodyObj.hospitalId = hospitalId;
        modifiedBody = JSON.stringify(bodyObj);
        
        // For debugging
        console.log(`Added hospital ID ${hospitalId} to request body`);
      } catch (e) {
        // If body is not valid JSON, leave it as is
        console.warn('Could not add hospitalId to request body:', e);
      }
    }
    
    return {
      ...options,
      headers,
      body: modifiedBody
    };
  }, [currentHospital, isSuperAdmin]);
  
  /**
   * Helper function for components to manually refresh their data when hospital changes
   * For use in useEffect: useEffect(() => { refreshData(fetchYourData); }, [refreshData]);
   */
  const refreshData = useCallback((fetchFunction: () => void) => {
    if (currentHospital) {
      console.log(`Refreshing data for hospital: ${currentHospital.name} (${currentHospital.id})`);
      fetchFunction();
    }
  }, [currentHospital]);

  return {
    currentHospital,
    loading,
    error,
    refetchHospitals,
    getSelectedHospitalId,
    filterByHospital,
    getHospitalUrlParam,
    getHospitalFetchOptions,
    refreshData,
    isSuperAdmin,
    isHospitalSelected: !!currentHospital,
    hospitalId: currentHospital?.id
  };
} 