import { useState, useEffect } from 'react';

export interface RegulatoryNews {
  id: string;
  title: string;
  source: string;
  date: string;
  category: string;
  content?: string | null;
  url?: string | null;
  isNew?: boolean;
}

export const useRegulatoryNews = () => {
  const [newsItems, setNewsItems] = useState<RegulatoryNews[]>([]);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  
  const fetchNews = async () => {
    setLoading(true);
    setError(null);
    
    try {
      console.log('Fetching regulatory news from API...');
      const response = await fetch('/api/regulations/news');
      
      console.log('News API response status:', response.status);
      
      if (!response.ok) {
        const errorText = await response.text();
        console.error('Error response from news API:', errorText);
        throw new Error(`Failed to fetch regulatory news: ${response.statusText || 'Unknown error'}`);
      }
      
      const data = await response.json();
      console.log('Successfully fetched news data, count:', data.length);
      setNewsItems(data);
    } catch (error) {
      console.error('Error fetching regulatory news:', error);
      setError(error instanceof Error ? error.message : 'An unknown error occurred');
      
      // Fallback to sample data in case of error
      console.log('Using fallback sample news data');
      setNewsItems([
        {
          id: 'rn1',
          title: 'JCI announces 7th Edition Standard updates',
          source: 'Joint Commission International',
          date: 'May 12, 2023',
          category: 'Standards Update',
          isNew: true,
        },
        {
          id: 'rn2',
          title: 'FDA releases new guidance on medical device reporting',
          source: 'U.S. FDA',
          date: 'May 10, 2023',
          category: 'Regulatory Guidance',
        },
        {
          id: 'rn3',
          title: 'GDPR enforcement actions increasing in healthcare sector',
          source: 'EU Commission',
          date: 'May 7, 2023',
          category: 'Enforcement',
        },
        {
          id: 'rn4',
          title: 'WHO publishes updated patient safety framework',
          source: 'World Health Organization',
          date: 'May 5, 2023',
          category: 'Standards Update',
          isNew: true,
        },
        {
          id: 'rn5',
          title: 'New patient privacy requirements announced',
          source: 'Regional Health Authority',
          date: 'May 9, 2023',
          category: 'Privacy',
          isNew: true,
        },
      ]);
    } finally {
      setLoading(false);
    }
  };
  
  useEffect(() => {
    fetchNews();
  }, []);
  
  // Create a new news item
  const createNews = async (newsData: Omit<RegulatoryNews, 'id'>) => {
    setLoading(true);
    
    try {
      console.log('Creating new regulatory news item:', newsData);
      const response = await fetch('/api/regulations/news', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(newsData),
      });
      
      if (!response.ok) {
        const errorText = await response.text();
        console.error('Error response from create news API:', errorText);
        throw new Error(`Failed to create regulatory news: ${response.statusText || 'Unknown error'}`);
      }
      
      await fetchNews(); // Refresh the list
      return true;
    } catch (error) {
      console.error('Error creating regulatory news:', error);
      setError(error instanceof Error ? error.message : 'Failed to create news');
      return false;
    } finally {
      setLoading(false);
    }
  };
  
  // Update a news item
  const updateNews = async (newsData: RegulatoryNews) => {
    setLoading(true);
    
    try {
      console.log('Updating regulatory news item:', newsData);
      const response = await fetch('/api/regulations/news', {
        method: 'PUT',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(newsData),
      });
      
      if (!response.ok) {
        const errorText = await response.text();
        console.error('Error response from update news API:', errorText);
        throw new Error(`Failed to update regulatory news: ${response.statusText || 'Unknown error'}`);
      }
      
      await fetchNews(); // Refresh the list
      return true;
    } catch (error) {
      console.error('Error updating regulatory news:', error);
      setError(error instanceof Error ? error.message : 'Failed to update news');
      return false;
    } finally {
      setLoading(false);
    }
  };
  
  // Delete a news item
  const deleteNews = async (id: string) => {
    setLoading(true);
    
    try {
      console.log('Deleting regulatory news item with ID:', id);
      const response = await fetch(`/api/regulations/news?id=${id}`, {
        method: 'DELETE',
      });
      
      if (!response.ok) {
        const errorText = await response.text();
        console.error('Error response from delete news API:', errorText);
        throw new Error(`Failed to delete regulatory news: ${response.statusText || 'Unknown error'}`);
      }
      
      await fetchNews(); // Refresh the list
      return true;
    } catch (error) {
      console.error('Error deleting regulatory news:', error);
      setError(error instanceof Error ? error.message : 'Failed to delete news');
      return false;
    } finally {
      setLoading(false);
    }
  };
  
  // Mark a news item as read (not new)
  const markAsRead = async (id: string) => {
    const newsItem = newsItems.find(item => item.id === id);
    
    if (!newsItem || !newsItem.isNew) {
      return true; // Already not new
    }
    
    return updateNews({
      ...newsItem,
      isNew: false,
    });
  };
  
  return {
    newsItems,
    loading,
    error,
    createNews,
    updateNews,
    deleteNews,
    markAsRead,
    refreshNews: fetchNews,
  };
}; 