import { useCallback, useEffect } from 'react'
import { useRouter } from 'next/navigation'
import { useSession } from 'next-auth/react'

interface NavigationItem {
  title: string
  path: string
  icon?: React.ComponentType
  alert?: boolean
  badge?: string
}

interface CacheItem {
  data: any
  timestamp: number
  ttl: number // Time to live in milliseconds
}

class NavigationCache {
  private cache = new Map<string, CacheItem>()
  private readonly DEFAULT_TTL = 5 * 60 * 1000 // 5 minutes

  set(key: string, data: any, ttl = this.DEFAULT_TTL) {
    this.cache.set(key, {
      data,
      timestamp: Date.now(),
      ttl
    })
  }

  get(key: string) {
    const item = this.cache.get(key)
    if (!item) return null

    // Check if item has expired
    if (Date.now() - item.timestamp > item.ttl) {
      this.cache.delete(key)
      return null
    }

    return item.data
  }

  clear() {
    this.cache.clear()
  }

  has(key: string) {
    const item = this.cache.get(key)
    if (!item) return false
    
    if (Date.now() - item.timestamp > item.ttl) {
      this.cache.delete(key)
      return false
    }
    
    return true
  }

  preload(key: string, fetcher: () => Promise<any>, ttl = this.DEFAULT_TTL) {
    // Don't preload if already cached and valid
    if (this.has(key)) return

    fetcher()
      .then(data => this.set(key, data, ttl))
      .catch(error => console.warn(`Failed to preload ${key}:`, error))
  }
}

const globalCache = new NavigationCache()

export const useNavigationCache = () => {
  const router = useRouter()
  const { data: session } = useSession()

  // Memoized user roles
  const userRoles = useCallback(() => ({
    isAdmin: session?.user?.role === 'admin' || session?.user?.role === 'super_admin',
    isSuperAdmin: session?.user?.role === 'super_admin'
  }), [session?.user?.role])

  // Memoized navigation items
  const navigationItems = useCallback(() => [
    { title: 'Dashboard', path: '/dashboard' },
    { title: 'Compliance', path: '/compliance', alert: true },
    { title: 'Findings', path: '/findings' },
    { title: 'Reports', path: '/reports' },
    { title: 'Documents', path: '/documents' },
    { title: 'Regulations', path: '/regulations' },
    { title: 'Schedule', path: '/schedule', alert: true },
    { title: 'Staff & Training', path: '/training' },
    { title: 'Departments', path: '/hospital/departments' },
  ], [])

  // Memoized admin items
  const adminItems = useCallback(() => [
    { title: 'Hospitals', path: '/admin/hospitals' },
    { title: 'All Departments', path: '/admin/departments' },
    { title: 'Settings', path: '/admin/settings' },
    { title: 'Security', path: '/admin/security', badge: 'New' },
    { title: 'Database', path: '/admin/database' },
    { title: 'Access', path: '/admin/access' },
  ], [])

  // Preload critical page data
  const preloadPageData = useCallback(async () => {
    // Preload dashboard data
    globalCache.preload('dashboard-metrics', async () => {
      const response = await fetch('/api/dashboard')
      if (!response.ok) throw new Error('Failed to fetch dashboard data')
      return response.json()
    })

    // Preload compliance data
    globalCache.preload('compliance-standards', async () => {
      const response = await fetch('/api/compliance/standards')
      if (!response.ok) throw new Error('Failed to fetch compliance standards')
      return response.json()
    })

    // Preload reports data
    globalCache.preload('reports-summary', async () => {
      const response = await fetch('/api/reports')
      if (!response.ok) throw new Error('Failed to fetch reports')
      return response.json()
    })

    // Preload recent findings
    globalCache.preload('recent-findings', async () => {
      const response = await fetch('/api/findings')
      if (!response.ok) throw new Error('Failed to fetch findings')
      return response.json()
    })

    // Preload documents categories
    globalCache.preload('document-categories', async () => {
      const response = await fetch('/api/documentation/categories')
      if (!response.ok) throw new Error('Failed to fetch document categories')
      return response.json()
    })

    // Preload regulatory news
    globalCache.preload('regulatory-news', async () => {
      const response = await fetch('/api/regulations/news')
      if (!response.ok) throw new Error('Failed to fetch regulatory news')
      return response.json()
    })

    // Preload training data
    globalCache.preload('training-data', async () => {
      const response = await fetch('/api/schedule/training')
      if (!response.ok) throw new Error('Failed to fetch training data')
      return response.json()
    })
  }, [])

  // Start preloading when the hook is used
  useEffect(() => {
    // Small delay to not block initial page render
    const timeoutId = setTimeout(preloadPageData, 100)
    return () => clearTimeout(timeoutId)
  }, [preloadPageData])

  // Enhanced navigation with cache clearing for related data
  const navigateToPage = useCallback((path: string) => {
    // Clear cache for pages that might have stale data
    if (path.includes('/dashboard')) {
      globalCache.clear()
      preloadPageData() // Refresh cache after navigation
    }
    
    router.push(path)
  }, [router, preloadPageData])

  return {
    cache: globalCache,
    navigateToPage,
    preloadPageData,
    getCachedData: globalCache.get.bind(globalCache),
    setCachedData: globalCache.set.bind(globalCache),
    clearCache: globalCache.clear.bind(globalCache)
  }
} 