import { useState, useEffect } from 'react';

interface FindingOption {
  id: number;
  value: string;
  label: string;
  description?: string;
  color_class?: string;
  sort_order: number;
}

interface UseFindingOptionsReturn {
  severities: FindingOption[];
  statuses: FindingOption[];
  severitiesLoading: boolean;
  statusesLoading: boolean;
  severitiesError: string | null;
  statusesError: string | null;
  getSeverityOptions: () => { value: string; label: string }[];
  getStatusOptions: () => { value: string; label: string }[];
  getSeverityColorClass: (value: string) => string;
  getStatusColorClass: (value: string) => string;
  refetch: () => void;
}

export const useFindingOptions = (): UseFindingOptionsReturn => {
  const [severities, setSeverities] = useState<FindingOption[]>([]);
  const [statuses, setStatuses] = useState<FindingOption[]>([]);
  const [severitiesLoading, setSeveritiesLoading] = useState(true);
  const [statusesLoading, setStatusesLoading] = useState(true);
  const [severitiesError, setSeveritiesError] = useState<string | null>(null);
  const [statusesError, setStatusesError] = useState<string | null>(null);

  const fetchSeverities = async () => {
    setSeveritiesLoading(true);
    setSeveritiesError(null);
    try {
      const response = await fetch('/api/findings/severities');
      if (!response.ok) {
        throw new Error('Failed to fetch severities');
      }
      const data = await response.json();
      setSeverities(data);
    } catch (error) {
      console.error('Error fetching severities:', error);
      setSeveritiesError(error instanceof Error ? error.message : 'Unknown error');
      // Fallback to static data
      setSeverities([
        { id: 1, value: 'critical', label: 'Critical', sort_order: 1, color_class: 'bg-red-700 text-white' },
        { id: 2, value: 'high', label: 'High', sort_order: 2, color_class: 'bg-red-500 text-white' },
        { id: 3, value: 'medium', label: 'Medium', sort_order: 3, color_class: 'bg-amber-500 text-white' },
        { id: 4, value: 'low', label: 'Low', sort_order: 4, color_class: 'bg-blue-500 text-white' },
      ]);
    } finally {
      setSeveritiesLoading(false);
    }
  };

  const fetchStatuses = async () => {
    setStatusesLoading(true);
    setStatusesError(null);
    try {
      const response = await fetch('/api/findings/statuses');
      if (!response.ok) {
        throw new Error('Failed to fetch statuses');
      }
      const data = await response.json();
      setStatuses(data);
    } catch (error) {
      console.error('Error fetching statuses:', error);
      setStatusesError(error instanceof Error ? error.message : 'Unknown error');
      // Fallback to static data
      setStatuses([
        { id: 1, value: 'open', label: 'Open', sort_order: 1, color_class: 'text-gray-600 border-gray-300' },
        { id: 2, value: 'in_progress', label: 'In Progress', sort_order: 2, color_class: 'text-blue-600 border-blue-300' },
        { id: 3, value: 'resolved', label: 'Resolved', sort_order: 3, color_class: 'text-green-600 border-green-300' },
      ]);
    } finally {
      setStatusesLoading(false);
    }
  };

  useEffect(() => {
    fetchSeverities();
    fetchStatuses();
  }, []);

  const getSeverityOptions = () => {
    return severities.map(severity => ({
      value: severity.value,
      label: severity.label
    }));
  };

  const getStatusOptions = () => {
    return statuses.map(status => ({
      value: status.value,
      label: status.label
    }));
  };

  const getSeverityColorClass = (value: string) => {
    const severity = severities.find(s => s.value === value);
    return severity?.color_class || 'bg-gray-500 text-white';
  };

  const getStatusColorClass = (value: string) => {
    const status = statuses.find(s => s.value === value);
    return status?.color_class || 'text-gray-600 border-gray-300';
  };

  const refetch = () => {
    fetchSeverities();
    fetchStatuses();
  };

  return {
    severities,
    statuses,
    severitiesLoading,
    statusesLoading,
    severitiesError,
    statusesError,
    getSeverityOptions,
    getStatusOptions,
    getSeverityColorClass,
    getStatusColorClass,
    refetch
  };
}; 