import { useState, useEffect, useCallback } from 'react'
import { toast } from 'sonner'

interface Department {
  id: string
  name: string
  description: string | null
  hospital_id: number | null
  createdAt: string
  updatedAt: string
}

export function useDepartments() {
  const [departments, setDepartments] = useState<Department[]>([])
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)

  const fetchDepartments = useCallback(async () => {
    setLoading(true)
    setError(null)
    
    try {
      console.log('Fetching departments...')
      const response = await fetch('/api/hospital/departments')
      
      if (!response.ok) {
        const errorText = await response.text()
        console.error('API error response:', errorText)
        throw new Error(`Failed to fetch departments: ${response.status} ${response.statusText}`)
      }
      
      const data = await response.json()
      console.log('Fetched departments:', data)
      
      // Handle the response format - it returns { departments: [...] }
      if (data.departments && Array.isArray(data.departments)) {
        setDepartments(data.departments)
      } else if (Array.isArray(data)) {
        setDepartments(data)
      } else {
        console.error('Unexpected response format:', data)
        setDepartments([])
      }
    } catch (err) {
      console.error('Error fetching departments:', err)
      setError(err instanceof Error ? err.message : 'An unknown error occurred')
    } finally {
      setLoading(false)
    }
  }, [])

  // Create a new department
  const createDepartment = async (data: {
    name: string
    description?: string
  }): Promise<boolean> => {
    try {
      const response = await fetch('/api/hospital/departments', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        },
        credentials: 'include',
        body: JSON.stringify(data)
      })

      if (!response.ok) {
        let errorMessage = 'Failed to create department'
        
        // Safely try to parse error response
        try {
          const responseText = await response.text()
          if (responseText) {
            const errorData = JSON.parse(responseText)
            errorMessage = errorData.message || errorMessage
          }
        } catch (parseError) {
          console.error('Error parsing error response:', parseError)
        }
        
        throw new Error(errorMessage)
      }

      await fetchDepartments()
      toast.success('Department created successfully')
      return true
    } catch (error) {
      console.error('Error creating department:', error)
      toast.error(error instanceof Error ? error.message : 'Failed to create department')
      return false
    }
  }

  // Update an existing department
  const updateDepartment = async (data: {
    id: string
    name?: string
    description?: string
  }): Promise<boolean> => {
    try {
      const response = await fetch('/api/hospital/departments', {
        method: 'PATCH',
        headers: {
          'Content-Type': 'application/json'
        },
        credentials: 'include',
        body: JSON.stringify(data)
      })

      if (!response.ok) {
        let errorMessage = 'Failed to update department'
        
        // Safely try to parse error response
        try {
          const responseText = await response.text()
          if (responseText) {
            const errorData = JSON.parse(responseText)
            errorMessage = errorData.message || errorMessage
          }
        } catch (parseError) {
          console.error('Error parsing error response:', parseError)
        }
        
        throw new Error(errorMessage)
      }

      await fetchDepartments()
      toast.success('Department updated successfully')
      return true
    } catch (error) {
      console.error('Error updating department:', error)
      toast.error(error instanceof Error ? error.message : 'Failed to update department')
      return false
    }
  }

  // Delete a department
  const deleteDepartment = async (id: string): Promise<boolean> => {
    try {
      const response = await fetch(`/api/hospital/departments?id=${id}`, {
        method: 'DELETE',
        credentials: 'include',
        headers: {
          'Content-Type': 'application/json'
        }
      })

      if (!response.ok) {
        let errorMessage = 'Failed to delete department'
        
        // Safely try to parse error response
        try {
          const responseText = await response.text()
          if (responseText) {
            const errorData = JSON.parse(responseText)
            errorMessage = errorData.message || errorMessage
          }
        } catch (parseError) {
          console.error('Error parsing error response:', parseError)
        }
        
        throw new Error(errorMessage)
      }

      await fetchDepartments()
      toast.success('Department deleted successfully')
      return true
    } catch (error) {
      console.error('Error deleting department:', error)
      toast.error(error instanceof Error ? error.message : 'Failed to delete department')
      return false
    }
  }

  // Load departments on component mount
  useEffect(() => {
    fetchDepartments()
  }, [fetchDepartments])

  return {
    departments,
    loading,
    error,
    refetch: fetchDepartments,
    // CRUD operations
    createDepartment,
    updateDepartment,
    deleteDepartment,
    // Helper functions
    getDepartmentName: (id: string) => {
      const department = departments.find(dept => dept.id === id)
      return department ? department.name : 'Unknown Department'
    },
    getDepartmentOptions: () => {
      // Create options for dropdown menus
      return departments.map(dept => ({
        value: dept.id,
        label: dept.name
      }))
    },
    // Function to convert department id to name or use fallback
    getDepartmentById: (departmentId: string | null | undefined) => {
      if (!departmentId) return null
      const found = departments.find(dept => dept.id === departmentId)
      return found
    }
  }
}

export default useDepartments 