import { useState, useEffect, useCallback } from 'react';
import { format } from 'date-fns';

export interface DashboardMetrics {
  auditCompletedCount: number;
  upcomingDeadlinesCount: number;
  staffCompliancePercentage: number;
  totalStandardsCount: number;
  complianceScore: number;
  previousComplianceScore: number;
  standardsData?: any[];
  trends?: {
    totalStandards: { value: number; isPositive: boolean; label: string; };
    auditsCompleted: { value: number; isPositive: boolean; label: string; };
    upcomingDeadlines: { value: number; isNeutral: boolean; label: string; };
    staffCompliance: { value: number; isPositive: boolean; label: string; };
  };
}

export interface ActionItem {
  id: string;
  title: string;
  dueDate: string;
  priority: string;
  assignee?: string;
  category: string;
  completed: boolean;
}

export interface ComplianceCard {
  title: string;
  description: string;
  standard: string;
  status: string;
  dueDate: string;
  compliance?: number;
  required?: number;
}

export interface RegulatoryNews {
  id: string;
  title: string;
  description: string;
  date: string;
  category: string;
  severity: string;
  source: string;
  link: string;
}

// Provide immediate default data to prevent blocking
const DEFAULT_METRICS: DashboardMetrics = {
  auditCompletedCount: 0,
  upcomingDeadlinesCount: 0,
          staffCompliancePercentage: 0,
          totalStandardsCount: 0,
          complianceScore: 82,
          previousComplianceScore: 78,
          standardsData: [
            { id: 1, name: 'ISO 9001', compliance: 87, required: 90 },
            { id: 2, name: 'HIPAA', compliance: 94, required: 90 },
            { id: 3, name: 'JCI', compliance: 78, required: 90 },
            { id: 4, name: 'GDPR', compliance: 82, required: 90 }
          ],
          trends: {
            totalStandards: { value: 5, isPositive: true, label: "from last month" },
    auditsCompleted: { value: 12, isPositive: true, label: "vs. last quarter" },
            upcomingDeadlines: { value: 2, isNeutral: true, label: "in next 14 days" },
            staffCompliance: { value: 3, isPositive: false, label: "below target" }
          }
}
        
const DEFAULT_COMPLIANCE_CARDS: ComplianceCard[] = [
          {
            title: "ISO 9001:2015 Quality Management",
            description: "Section 4.2 - Documentation Requirements",
            standard: "ISO 9001",
            status: "at-risk",
            dueDate: "2024-06-15",
            compliance: 87,
            required: 90
          },
          {
            title: "HIPAA Security Rule",
            description: "§164.308(a)(5)(i) - Security Awareness Training",
            standard: "HIPAA",
            status: "compliant",
            dueDate: "2024-07-30",
            compliance: 94,
            required: 90
          },
          {
            title: "Joint Commission Patient Safety",
            description: "NPSG.03.05.01 - Medication Reconciliation",
            standard: "JCI",
            status: "non-compliant",
            dueDate: "2024-06-10",
            compliance: 78,
            required: 90
          }
]
        
const DEFAULT_ACTION_ITEMS: ActionItem[] = [
          {
            id: "a1",
            title: "Review updated HIPAA documentation requirements",
            dueDate: "2024-05-25",
            priority: "high",
            assignee: "Compliance Team",
            category: "compliance",
            completed: false
          },
          {
            id: "a2",
            title: "Schedule ISO 9001 internal audit follow-up",
            dueDate: "2024-06-05",
            priority: "medium",
            assignee: "Quality Manager",
            category: "audit",
            completed: false
          },
          {
            id: "a3",
            title: "Update staff emergency response training materials",
            dueDate: "2024-06-15",
            priority: "low",
            assignee: "Training Dept.",
            category: "training",
            completed: true
          }
]
        
const DEFAULT_REGULATORY_NEWS: RegulatoryNews[] = [
          {
            id: 'news-1',
            title: 'HIPAA Privacy Rule Amendments',
            description: 'New patient data sharing requirements coming into effect July 2023.',
            date: new Date().toISOString(),
            category: 'regulation',
            severity: 'info',
            source: 'HHS',
            link: '#'
          },
          {
            id: 'news-2',
            title: 'Joint Commission Patient Safety Goals',
            description: 'Updated criteria for medication reconciliation processes.',
            date: new Date().toISOString(),
            category: 'safety',
            severity: 'warning',
            source: 'Joint Commission',
            link: '#'
          },
          {
            id: 'news-3',
            title: 'FDA Guidance on Medical Device Security',
            description: 'New cybersecurity requirements for connected medical devices.',
            date: new Date().toISOString(),
            category: 'security',
            severity: 'info',
            source: 'FDA',
            link: '#'
          }
]

export const useDashboardData = () => {
  // Start with default data for immediate render
  const [metrics, setMetrics] = useState<DashboardMetrics>(DEFAULT_METRICS);
  const [recentFindings, setRecentFindings] = useState<any[]>([]);
  const [complianceCards, setComplianceCards] = useState<ComplianceCard[]>(DEFAULT_COMPLIANCE_CARDS);
  const [actionItems, setActionItems] = useState<ActionItem[]>(DEFAULT_ACTION_ITEMS);
  const [regulatoryNews, setRegulatoryNews] = useState<RegulatoryNews[]>(DEFAULT_REGULATORY_NEWS);
  const [loading, setLoading] = useState(false); // Start as false for immediate render
  const [error, setError] = useState<string | null>(null);

  // Async data fetcher - runs in background
  const fetchDashboardData = useCallback(async () => {
    try {
      setLoading(true);
      console.log('🔄 Fetching dashboard data...');
      
      const response = await fetch('/api/dashboard');
      
      if (!response.ok) {
        const errorData = await response.json();
        console.error('❌ Dashboard API error:', errorData);
        throw new Error(errorData.error || 'Failed to fetch dashboard data');
      }
      
      const data = await response.json();
      console.log('✅ Dashboard data received:', data);
      console.log('📊 Staff compliance from API:', data.metrics?.staffCompliancePercentage);
      
      // Update state with real data
      setMetrics(data.metrics);
      setRecentFindings(data.recentFindings || []);
      setComplianceCards(data.complianceCards || DEFAULT_COMPLIANCE_CARDS);
      setActionItems(data.actionItems || DEFAULT_ACTION_ITEMS);
      setRegulatoryNews(data.regulatoryNews || DEFAULT_REGULATORY_NEWS);
      setError(null);
      
      console.log('🎯 Updated metrics state with staff compliance:', data.metrics?.staffCompliancePercentage);
    } catch (err) {
      console.error('❌ Error fetching dashboard data:', err);
      setError((err as Error).message);
      // Keep default data on error
      } finally {
        setLoading(false);
      }
  }, [])

  useEffect(() => {
    // Defer data fetching to not block navigation
    const timeoutId = setTimeout(() => {
    fetchDashboardData();
    }, 100); // Small delay to allow page to render first

    return () => clearTimeout(timeoutId);
  }, [fetchDashboardData]);

  return { 
    metrics, 
    recentFindings, 
    complianceCards, 
    actionItems, 
    regulatoryNews, 
    loading, 
    error,
    refetch: fetchDashboardData
  };
}; 