import { useState, useEffect } from 'react';
import { format } from 'date-fns';

export interface AuditFinding {
  id: number;
  title: string;
  department: string;
  severity: 'critical' | 'high' | 'medium' | 'low';
  status: 'open' | 'in_progress' | 'resolved';
  description: string | null;
  date_reported: string;
  resolution_date?: string | null;
  timeline?: string | null;
  hospital_id: number;
  created_at: string;
  updated_at: string;
}

export const useAuditFindings = () => {
  const [findings, setFindings] = useState<AuditFinding[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [deleting, setDeleting] = useState(false);
  const [updating, setUpdating] = useState(false);
  
  const fetchData = async () => {
    setLoading(true);
    try {
      const response = await fetch('/api/findings');
      
      if (!response.ok) {
        throw new Error('Failed to fetch audit findings');
      }
      
      const data = await response.json();
      
      // Format the findings for display
      const formattedFindings = data.map((finding: AuditFinding) => ({
        ...finding,
        // Convert date to readable format
        date: format(new Date(finding.date_reported), 'MMM dd, yyyy')
      }));
      
      setFindings(formattedFindings);
      setError(null);
    } catch (err) {
      console.error('Error fetching audit findings:', err);
      setError((err as Error).message);
    } finally {
      setLoading(false);
    }
  };
  
  useEffect(() => {
    fetchData();
  }, []);
  
  const deleteFinding = async (id: number) => {
    setDeleting(true);
    try {
      const response = await fetch(`/api/findings/${id}`, {
        method: 'DELETE',
      });
      
      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.error || 'Failed to delete finding');
      }
      
      // Remove the finding from the local state
      setFindings(findings.filter(finding => finding.id !== id));
      return true;
    } catch (err) {
      console.error('Error deleting finding:', err);
      setError((err as Error).message);
      return false;
    } finally {
      setDeleting(false);
    }
  };
  
  const updateFinding = async (id: number, updatedData: Partial<AuditFinding>) => {
    setUpdating(true);
    try {
      const response = await fetch(`/api/findings/${id}`, {
        method: 'PUT',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(updatedData),
      });
      
      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.error || 'Failed to update finding');
      }
      
      const updatedFinding = await response.json();
      
      // Update the finding in the local state
      setFindings(findings.map(finding => 
        finding.id === id 
          ? { 
              ...updatedFinding, 
              date: format(new Date(updatedFinding.date_reported), 'MMM dd, yyyy') 
            } 
          : finding
      ));
      
      return true;
    } catch (err) {
      console.error('Error updating finding:', err);
      setError((err as Error).message);
      return false;
    } finally {
      setUpdating(false);
    }
  };
  
  const refetch = () => {
    fetchData();
  };
  
  return { 
    findings, 
    loading, 
    error, 
    refetch, 
    deleteFinding, 
    updateFinding, 
    deleting,
    updating 
  };
}; 