#!/bin/bash

# Export database for the audit-next Next.js project
# This script exports only the health_audit database that this project uses

PROJECT_NAME="audit-next"
DB_NAME="health_audit"
DB_HOST="localhost"
DB_PORT="3306"
DB_USER="root"
EXPORT_FILE="${PROJECT_NAME}_database.sql"

echo "🚀 Exporting database for Next.js project: $PROJECT_NAME"
echo "📊 Target Database: $DB_NAME"
echo "🖥️  Host: $DB_HOST:$DB_PORT"
echo "📁 Export file: $EXPORT_FILE"
echo ""

# Verify this is the correct database for the project
echo "🔍 Verifying database connection..."
if ! npx prisma migrate status > /dev/null 2>&1; then
    echo "❌ Error: Cannot verify database connection through Prisma"
    echo "   Make sure you're in the correct Next.js project directory"
    exit 1
fi

# Get the actual database name from Prisma
PRISMA_DB_INFO=$(npx prisma migrate status 2>/dev/null | grep "Datasource")
echo "✅ Confirmed: $PRISMA_DB_INFO"
echo ""

# Export only the tables that belong to this Next.js project
echo "⏳ Exporting project-specific database..."
mysqldump \
    --host=$DB_HOST \
    --port=$DB_PORT \
    --user=$DB_USER \
    --single-transaction \
    --add-drop-table \
    --create-options \
    --disable-keys \
    --extended-insert \
    --set-charset \
    --comments \
    --databases $DB_NAME > $EXPORT_FILE 2>/dev/null

# Check if export was successful
if [ $? -eq 0 ] && [ -f "$EXPORT_FILE" ] && [ -s "$EXPORT_FILE" ]; then
    FILE_SIZE=$(du -h $EXPORT_FILE | cut -f1)
    TABLE_COUNT=$(grep -c "CREATE TABLE" $EXPORT_FILE)
    
    echo ""
    echo "✅ Project database exported successfully!"
    echo "📁 File: $EXPORT_FILE"
    echo "📊 Size: $FILE_SIZE"
    echo "📋 Tables: $TABLE_COUNT"
    echo ""
    echo "🎯 This export contains:"
    echo "  ✓ Complete $DB_NAME database structure"
    echo "  ✓ All data from your Next.js project"
    echo "  ✓ User accounts and authentication"
    echo "  ✓ Hospital and compliance data"
    echo "  ✓ All project-specific tables and relations"
    echo ""
    echo "🔄 To restore on another system:"
    echo "   mysql -u root < $EXPORT_FILE"
    echo ""
    echo "📋 Project tables included:"
    grep "CREATE TABLE" $EXPORT_FILE | sed 's/CREATE TABLE `/  - /' | sed 's/` (//' | head -10
    if [ $TABLE_COUNT -gt 10 ]; then
        echo "  ... and $(($TABLE_COUNT - 10)) more tables"
    fi
    
else
    echo ""
    echo "❌ Database export failed!"
    echo "🔍 Please check:"
    echo "  - MySQL server is running"
    echo "  - You're in the correct project directory"
    echo "  - Database connection is working"
    echo "  - User has proper permissions"
    
    # Clean up failed export file
    if [ -f "$EXPORT_FILE" ]; then
        rm "$EXPORT_FILE"
        echo "  - Cleaned up incomplete export file"
    fi
    
    exit 1
fi 