import React, { useState } from 'react';
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { PlusCircle, Edit, Trash2 } from "lucide-react";
import { 
  Dialog, 
  DialogContent, 
  DialogDescription, 
  DialogFooter, 
  DialogHeader, 
  DialogTitle, 
  DialogTrigger,
  DialogClose 
} from "@/components/ui/dialog";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
  AlertDialogTrigger,
} from "@/components/ui/alert-dialog";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Checkbox } from "@/components/ui/checkbox";
import { TrainingEvent } from '@/hooks/useTrainingData';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import useDepartments from '@/hooks/useDepartments';
import { toast } from "sonner";

interface TrainingEventListProps {
  events: TrainingEvent[];
  loading: boolean;
  error: string | null;
  onAdd: (event: Omit<TrainingEvent, 'id'>) => Promise<boolean>;
  onEdit: (event: TrainingEvent) => Promise<boolean>;
  onDelete: (id: string) => Promise<boolean>;
}

export function TrainingEventList({
  events,
  loading,
  error,
  onAdd,
  onEdit,
  onDelete
}: TrainingEventListProps) {
  const [selectedEvent, setSelectedEvent] = useState<TrainingEvent | null>(null);
  const [isAddDialogOpen, setIsAddDialogOpen] = useState(false);
  const [isEditDialogOpen, setIsEditDialogOpen] = useState(false);
  const [isDeleteDialogOpen, setIsDeleteDialogOpen] = useState(false);
  
  // Move the useDepartments hook to the top level
  const { getDepartmentOptions, loading: loadingDepts } = useDepartments();
  
  // Form state for add/edit
  const [formData, setFormData] = useState<Omit<TrainingEvent, 'id'>>({
    title: '',
    date: '',
    instructor: '',
    location: '',
    department: '',
    duration: '',
    capacity: 0,
    enrolled: 0,
    required: false
  });
  
  // Reset form data
  const resetFormData = () => {
    setFormData({
      title: '',
      date: '',
      instructor: '',
      location: '',
      department: '',
      duration: '',
      capacity: 0,
      enrolled: 0,
      required: false
    });
  };
  
  // Open edit dialog with selected event data
  const handleEditClick = (event: TrainingEvent) => {
    setSelectedEvent(event);
    setFormData({
      title: event.title,
      date: event.date,
      instructor: event.instructor,
      location: event.location,
      department: event.department,
      duration: event.duration,
      capacity: event.capacity,
      enrolled: event.enrolled,
      required: event.required
    });
    setIsEditDialogOpen(true);
  };
  
  // Open delete dialog with selected event
  const handleDeleteClick = (event: TrainingEvent) => {
    setSelectedEvent(event);
    setIsDeleteDialogOpen(true);
  };
  
  // Handle form input changes
  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const { name, value, type } = e.target;
    if (type === 'number') {
      setFormData(prev => ({
        ...prev,
        [name]: parseInt(value) || 0
      }));
    } else {
      setFormData(prev => ({
        ...prev,
        [name]: value
      }));
    }
  };
  
  // Handle checkbox changes
  const handleCheckboxChange = (checked: boolean) => {
    setFormData(prev => ({
      ...prev,
      required: checked
    }));
  };
  
  // Handle form submission for adding a new event
  const handleAddSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    console.log('=== Training Form Submit - Start ===');
    console.log('Raw form data:', formData);
    
    // Validate form data before submission
    console.log('Validating form data...');
    
    const validationErrors = [];
    if (!formData.title?.trim()) validationErrors.push('title is empty');
    if (!formData.date) validationErrors.push('date is empty');
    if (!formData.instructor?.trim()) validationErrors.push('instructor is empty');
    if (!formData.location?.trim()) validationErrors.push('location is empty');
    if (!formData.department) validationErrors.push('department is empty');
    if (!formData.duration?.trim()) validationErrors.push('duration is empty');
    if (formData.capacity <= 0) validationErrors.push(`capacity is invalid: ${formData.capacity}`);
    if (formData.enrolled < 0) validationErrors.push(`enrolled is invalid: ${formData.enrolled}`);
    
    if (validationErrors.length > 0) {
      console.error('Validation errors:', validationErrors);
      toast.error(`Validation failed: ${validationErrors.join(', ')}`);
      return;
    }
    
    console.log('Validation passed, preparing cleaned data...');
    
    // Prepare cleaned data for submission
    const cleanedData = {
      title: formData.title.trim(),
      date: formData.date,
      instructor: formData.instructor.trim(),
      location: formData.location.trim(),
      department: formData.department,
      duration: formData.duration.trim(),
      capacity: Number(formData.capacity),
      enrolled: Number(formData.enrolled),
      required: formData.required
    };
    
    console.log('Cleaned data for submission:', cleanedData);
    console.log('Data types:', {
      title: typeof cleanedData.title,
      date: typeof cleanedData.date,
      instructor: typeof cleanedData.instructor,
      location: typeof cleanedData.location,
      department: typeof cleanedData.department,
      duration: typeof cleanedData.duration,
      capacity: typeof cleanedData.capacity,
      enrolled: typeof cleanedData.enrolled,
      required: typeof cleanedData.required
    });
    
    console.log('Calling onAdd function...');
    const success = await onAdd(cleanedData);
    console.log('onAdd result:', success);
    
    if (success) {
      console.log('Success! Closing dialog and resetting form...');
      setIsAddDialogOpen(false);
      resetFormData();
    } else {
      console.error('onAdd returned false - submission failed');
    }
    
    console.log('=== Training Form Submit - End ===');
  };
  
  // Handle form submission for editing an event
  const handleEditSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!selectedEvent) return;
    
    // Validate form data before submission
    if (!formData.title.trim() || !formData.date || !formData.instructor.trim() || 
        !formData.location.trim() || !formData.department || !formData.duration.trim() ||
        formData.capacity <= 0 || formData.enrolled < 0) {
      toast.error('Please fill in all required fields with valid values');
      return;
    }
    
    // Prepare cleaned data for submission
    const cleanedData = {
      id: selectedEvent.id,
      title: formData.title.trim(),
      date: formData.date,
      instructor: formData.instructor.trim(),
      location: formData.location.trim(),
      department: formData.department,
      duration: formData.duration.trim(),
      capacity: Number(formData.capacity),
      enrolled: Number(formData.enrolled),
      required: formData.required
    };
    
    const success = await onEdit(cleanedData);
    
    if (success) {
      setIsEditDialogOpen(false);
      setSelectedEvent(null);
      resetFormData();
    }
  };
  
  // Handle confirmation of delete
  const handleDeleteConfirm = async () => {
    if (!selectedEvent) return;
    
    const success = await onDelete(selectedEvent.id);
    if (success) {
      setIsDeleteDialogOpen(false);
      setSelectedEvent(null);
    }
  };
  
  // Loading state
  if (loading) {
    return (
      <div className="flex justify-center items-center h-64">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-gray-900"></div>
      </div>
    );
  }
  
  // Error state
  if (error) {
    return (
      <div className="flex justify-center items-center h-64">
        <div className="text-red-500">Error: {error}</div>
      </div>
    );
  }
  
  // Empty state
  if (events.length === 0) {
    return (
      <div className="text-center py-12">
        <h3 className="text-lg font-medium mb-2">No training events found</h3>
        <p className="text-muted-foreground mb-6">Get started by creating your first training event.</p>
        <Dialog open={isAddDialogOpen} onOpenChange={setIsAddDialogOpen}>
          <DialogTrigger asChild>
            <Button>
              <PlusCircle className="h-4 w-4 mr-2" />
              Add Training Event
            </Button>
          </DialogTrigger>
          {renderAddEditForm("add")}
        </Dialog>
      </div>
    );
  }
  
  return (
    <div className="space-y-4">
      <div className="flex justify-between items-center">
        <h2 className="text-xl font-semibold">Training Events</h2>
        <Dialog open={isAddDialogOpen} onOpenChange={setIsAddDialogOpen}>
          <DialogTrigger asChild>
            <Button>
              <PlusCircle className="h-4 w-4 mr-2" />
              Add Event
            </Button>
          </DialogTrigger>
          {renderAddEditForm("add")}
        </Dialog>
      </div>
      
      <div className="border rounded-md">
        <Table>
          <TableHeader>
            <TableRow>
              <TableHead>Title</TableHead>
              <TableHead>Date</TableHead>
              <TableHead>Instructor</TableHead>
              <TableHead>Department</TableHead>
              <TableHead>Location</TableHead>
              <TableHead>Duration</TableHead>
              <TableHead className="text-center">Required</TableHead>
              <TableHead className="text-center">Capacity</TableHead>
              <TableHead className="text-right">Actions</TableHead>
            </TableRow>
          </TableHeader>
          <TableBody>
            {events.map((event) => (
              <TableRow key={event.id}>
                <TableCell className="font-medium">{event.title}</TableCell>
                <TableCell>{formatDate(event.date)}</TableCell>
                <TableCell>{event.instructor}</TableCell>
                <TableCell>{event.department}</TableCell>
                <TableCell>{event.location}</TableCell>
                <TableCell>{event.duration}</TableCell>
                <TableCell className="text-center">
                  {event.required ? (
                    <Badge variant="outline" className="bg-blue-50 text-blue-700 border-blue-200">Required</Badge>
                  ) : (
                    <Badge variant="outline" className="bg-gray-50 text-gray-500 border-gray-200">Optional</Badge>
                  )}
                </TableCell>
                <TableCell className="text-center">
                  {event.enrolled}/{event.capacity}
                </TableCell>
                <TableCell className="text-right">
                  <div className="flex justify-end space-x-2">
                    <Button variant="ghost" size="icon" onClick={() => handleEditClick(event)}>
                      <Edit className="h-4 w-4" />
                    </Button>
                    <Button variant="ghost" size="icon" onClick={() => handleDeleteClick(event)}>
                      <Trash2 className="h-4 w-4" />
                    </Button>
                  </div>
                </TableCell>
              </TableRow>
            ))}
          </TableBody>
        </Table>
      </div>
      
      {/* Edit Dialog */}
      <Dialog open={isEditDialogOpen} onOpenChange={setIsEditDialogOpen}>
        {renderAddEditForm("edit")}
      </Dialog>
      
      {/* Delete Confirmation */}
      <AlertDialog open={isDeleteDialogOpen} onOpenChange={setIsDeleteDialogOpen}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>Delete Training Event</AlertDialogTitle>
            <AlertDialogDescription>
              Are you sure you want to delete this training event? This action cannot be undone.
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel>Cancel</AlertDialogCancel>
            <AlertDialogAction onClick={handleDeleteConfirm} className="bg-red-600 hover:bg-red-700">
              Delete
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </div>
  );

  // Helper function to render add/edit form
  function renderAddEditForm(type: "add" | "edit") {
    const departmentOptions = loadingDepts ? [
      { value: 'Nursing', label: 'Nursing' },
      { value: 'Pharmacy', label: 'Pharmacy' },
      { value: 'Laboratory', label: 'Laboratory' }
    ] : getDepartmentOptions();
    
    return (
      <DialogContent className="sm:max-w-[425px]">
        <DialogHeader>
          <DialogTitle>{type === 'add' ? 'Add New Training Event' : 'Edit Training Event'}</DialogTitle>
          <DialogDescription>
            {type === 'add' 
              ? 'Create a new training event for your staff.' 
              : 'Update the details of this training event.'}
          </DialogDescription>
        </DialogHeader>
        <form onSubmit={type === 'add' ? handleAddSubmit : handleEditSubmit}>
          <div className="grid gap-4 py-4">
            <div className="grid grid-cols-4 items-center gap-4">
              <Label htmlFor="title" className="text-right">
                Title
              </Label>
              <Input
                id="title"
                name="title"
                value={formData.title}
                onChange={handleInputChange}
                className="col-span-3"
                required
              />
            </div>
            <div className="grid grid-cols-4 items-center gap-4">
              <Label htmlFor="date" className="text-right">
                Date
              </Label>
              <Input
                id="date"
                name="date"
                type="date"
                value={formData.date || ''}
                onChange={handleInputChange}
                className="col-span-3"
                required
              />
            </div>
            <div className="grid grid-cols-4 items-center gap-4">
              <Label htmlFor="instructor" className="text-right">
                Instructor
              </Label>
              <Input
                id="instructor"
                name="instructor"
                value={formData.instructor}
                onChange={handleInputChange}
                className="col-span-3"
                required
              />
            </div>
            <div className="grid grid-cols-4 items-center gap-4">
              <Label htmlFor="department" className="text-right">
                Department
              </Label>
              <div className="col-span-3">
                <Select 
                  value={formData.department} 
                  onValueChange={(value) => setFormData(prev => ({ ...prev, department: value }))}
                >
                  <SelectTrigger className="w-full">
                    <SelectValue placeholder="Select Department" />
                  </SelectTrigger>
                  <SelectContent>
                    {departmentOptions.map(department => (
                      <SelectItem key={department.value} value={department.value}>
                        {department.label}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
            </div>
            <div className="grid grid-cols-4 items-center gap-4">
              <Label htmlFor="location" className="text-right">
                Location
              </Label>
              <Input
                id="location"
                name="location"
                value={formData.location}
                onChange={handleInputChange}
                className="col-span-3"
                required
              />
            </div>
            <div className="grid grid-cols-4 items-center gap-4">
              <Label htmlFor="duration" className="text-right">
                Duration
              </Label>
              <Input
                id="duration"
                name="duration"
                value={formData.duration}
                onChange={handleInputChange}
                className="col-span-3"
                required
                placeholder="e.g. 2 hours"
              />
            </div>
            <div className="grid grid-cols-4 items-center gap-4">
              <Label htmlFor="capacity" className="text-right">
                Capacity
              </Label>
              <Input
                id="capacity"
                name="capacity"
                type="number"
                min="1"
                value={formData.capacity}
                onChange={handleInputChange}
                className="col-span-3"
                required
              />
            </div>
            <div className="grid grid-cols-4 items-center gap-4">
              <Label htmlFor="enrolled" className="text-right">
                Enrolled
              </Label>
              <Input
                id="enrolled"
                name="enrolled"
                type="number"
                min="0"
                value={formData.enrolled}
                onChange={handleInputChange}
                className="col-span-3"
                required
              />
            </div>
            <div className="grid grid-cols-4 items-center gap-4">
              <Label htmlFor="required" className="text-right">
                Required
              </Label>
              <div className="flex items-center space-x-2 col-span-3">
                <Checkbox 
                  id="required" 
                  checked={formData.required} 
                  onCheckedChange={handleCheckboxChange} 
                />
                <label
                  htmlFor="required"
                  className="text-sm font-medium leading-none peer-disabled:cursor-not-allowed peer-disabled:opacity-70"
                >
                  Mandatory training
                </label>
              </div>
            </div>
          </div>
          <DialogFooter>
            <Button type="submit">{type === 'add' ? 'Add Event' : 'Save Changes'}</Button>
          </DialogFooter>
        </form>
      </DialogContent>
    );
  }
}

// Helper function to format date
function formatDate(dateString: string): string {
  try {
    const date = new Date(dateString);
    return date.toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
    });
  } catch (e) {
    return dateString;
  }
} 