import React, { useState } from 'react';
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Progress } from "@/components/ui/progress";
import { Button } from "@/components/ui/button";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogFooter, DialogTrigger } from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { AlertDialog, AlertDialogAction, AlertDialogCancel, AlertDialogContent, AlertDialogDescription, AlertDialogFooter, AlertDialogHeader, AlertDialogTitle } from "@/components/ui/alert-dialog";
import { Pencil, Trash2, Plus, Loader2 } from "lucide-react";

export interface DepartmentCompliance {
  id?: number;
  name: string;
  complianceRate: number;
  staffCount: number;
  lastUpdated?: string | Date;
  notes?: string;
}

interface StaffComplianceTableProps {
  overallRate: number;
  departments: DepartmentCompliance[];
  loading: boolean;
  onAdd?: (data: Omit<DepartmentCompliance, 'id'>) => Promise<boolean>;
  onEdit?: (data: DepartmentCompliance) => Promise<boolean>;
  onDelete?: (id: number) => Promise<boolean>;
}

export function StaffComplianceTable({
  overallRate,
  departments,
  loading,
  onAdd,
  onEdit,
  onDelete
}: StaffComplianceTableProps) {
  const [isAddDialogOpen, setIsAddDialogOpen] = useState(false);
  const [isEditDialogOpen, setIsEditDialogOpen] = useState(false);
  const [isDeleteDialogOpen, setIsDeleteDialogOpen] = useState(false);
  const [currentDept, setCurrentDept] = useState<DepartmentCompliance | null>(null);
  const [formData, setFormData] = useState({
    department: '',
    staffCount: '',
    complianceRate: '',
    notes: ''
  });
  const [isSubmitting, setIsSubmitting] = useState(false);
  
  // Handle form input changes
  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) => {
    const { name, value } = e.target;
    setFormData(prev => ({
      ...prev,
      [name]: value
    }));
  };
  
  // Handle add/edit form submission
  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setIsSubmitting(true);
    
    const data = {
      department: formData.department,
      staffCount: parseInt(formData.staffCount, 10),
      complianceRate: parseInt(formData.complianceRate, 10),
      notes: formData.notes
    };
    
    console.log('Submitting staff compliance form data:', data);
    
    try {
      let success = false;
      
      if (currentDept && currentDept.id && onEdit) {
        // Edit existing record
        console.log('Updating existing department:', currentDept.id);
        success = await onEdit({
          id: currentDept.id,
          name: data.department,
          staffCount: data.staffCount,
          complianceRate: data.complianceRate
        });
      } else if (onAdd) {
        // Add new record
        console.log('Adding new department');
        success = await onAdd({
          name: data.department,
          staffCount: data.staffCount,
          complianceRate: data.complianceRate,
          notes: data.notes
        });
      }
      
      console.log('Form submission result:', success);
      
      if (success) {
        // Close dialog and reset form
        setIsAddDialogOpen(false);
        setIsEditDialogOpen(false);
        resetForm();
      }
    } catch (error) {
      console.error("Error submitting form:", error);
    } finally {
      setIsSubmitting(false);
    }
  };
  
  // Handle edit button click
  const handleEditClick = (dept: DepartmentCompliance) => {
    setCurrentDept(dept);
    setFormData({
      department: dept.name,
      staffCount: dept.staffCount.toString(),
      complianceRate: dept.complianceRate.toString(),
      notes: ''
    });
    setIsEditDialogOpen(true);
  };
  
  // Handle delete button click
  const handleDeleteClick = (dept: DepartmentCompliance) => {
    setCurrentDept(dept);
    setIsDeleteDialogOpen(true);
  };
  
  // Handle delete confirmation
  const handleDeleteConfirm = async () => {
    if (currentDept?.id && onDelete) {
      setIsSubmitting(true);
      try {
        const success = await onDelete(currentDept.id);
        if (success) {
          setIsDeleteDialogOpen(false);
        }
      } catch (error) {
        console.error("Error deleting record:", error);
      } finally {
        setIsSubmitting(false);
      }
    }
  };
  
  // Reset form data
  const resetForm = () => {
    setFormData({
      department: '',
      staffCount: '',
      complianceRate: '',
      notes: ''
    });
    setCurrentDept(null);
  };
  
  // Open add dialog
  const openAddDialog = () => {
    resetForm();
    setIsAddDialogOpen(true);
  };
  
  if (loading) {
    return (
      <div className="flex justify-center items-center h-64">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-gray-900"></div>
      </div>
    );
  }
  
  return (
    <Card>
      <CardHeader>
        <div className="flex justify-between items-center">
          <CardTitle>Staff Compliance Overview</CardTitle>
          <div className="flex items-center gap-4">
            <div className="text-right">
              <span className="text-sm text-muted-foreground">Overall Compliance</span>
              <div className="text-2xl font-bold">{overallRate}%</div>
            </div>
            {onAdd && (
              <Button 
                onClick={openAddDialog} 
                className="flex items-center gap-1"
              >
                <Plus className="h-4 w-4" />
                Add Department
              </Button>
            )}
          </div>
        </div>
      </CardHeader>
      <CardContent>
        <div className="space-y-6">
          <div>
            <div className="flex justify-between mb-2">
              <span className="text-sm font-medium">Hospital-wide Training Completion</span>
              <span className="text-sm">{overallRate}%</span>
            </div>
            <Progress value={overallRate} className="h-2" />
          </div>
          
          <div className="border rounded-md">
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>Department</TableHead>
                  <TableHead>Staff Count</TableHead>
                  <TableHead>Compliance Rate</TableHead>
                  <TableHead className="w-[300px]">Progress</TableHead>
                  {(onEdit || onDelete) && (
                    <TableHead className="w-[100px] text-right">Actions</TableHead>
                  )}
                </TableRow>
              </TableHeader>
              <TableBody>
                {departments.length === 0 ? (
                  <TableRow>
                    <TableCell colSpan={onEdit || onDelete ? 5 : 4} className="text-center text-muted-foreground py-6">
                      No staff compliance data available. Add a department to get started.
                    </TableCell>
                  </TableRow>
                ) : (
                  departments.map((dept) => (
                    <TableRow key={dept.id || dept.name}>
                      <TableCell className="font-medium">{dept.name}</TableCell>
                      <TableCell>{dept.staffCount}</TableCell>
                      <TableCell>{dept.complianceRate}%</TableCell>
                      <TableCell>
                        <div className="flex items-center gap-2">
                          <Progress value={dept.complianceRate} className="h-2" />
                          <span className="text-sm">{getComplianceStatus(dept.complianceRate)}</span>
                        </div>
                      </TableCell>
                      {(onEdit || onDelete) && (
                        <TableCell className="text-right">
                          <div className="flex justify-end gap-2">
                            {onEdit && (
                              <Button 
                                variant="ghost" 
                                size="icon" 
                                onClick={() => handleEditClick(dept)}
                                title="Edit"
                              >
                                <Pencil className="h-4 w-4" />
                              </Button>
                            )}
                            {onDelete && (
                              <Button 
                                variant="ghost" 
                                size="icon"
                                onClick={() => handleDeleteClick(dept)}
                                title="Delete"
                                className="text-red-500 hover:text-red-700 hover:bg-red-50"
                              >
                                <Trash2 className="h-4 w-4" />
                              </Button>
                            )}
                          </div>
                        </TableCell>
                      )}
                    </TableRow>
                  ))
                )}
              </TableBody>
            </Table>
          </div>
        </div>
      </CardContent>
      
      {/* Add Department Dialog */}
      <Dialog open={isAddDialogOpen} onOpenChange={setIsAddDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Add Department Compliance</DialogTitle>
          </DialogHeader>
          <form onSubmit={handleSubmit}>
            <div className="grid gap-4 py-4">
              <div className="grid gap-2">
                <Label htmlFor="department">Department Name</Label>
                <Input 
                  id="department"
                  name="department"
                  value={formData.department}
                  onChange={handleInputChange}
                  placeholder="e.g., Nursing"
                  required
                />
              </div>
              <div className="grid grid-cols-2 gap-4">
                <div className="grid gap-2">
                  <Label htmlFor="staffCount">Staff Count</Label>
                  <Input 
                    id="staffCount"
                    name="staffCount"
                    type="number"
                    min="1"
                    value={formData.staffCount}
                    onChange={handleInputChange}
                    placeholder="e.g., 45"
                    required
                  />
                </div>
                <div className="grid gap-2">
                  <Label htmlFor="complianceRate">Compliance Rate (%)</Label>
                  <Input 
                    id="complianceRate"
                    name="complianceRate"
                    type="number"
                    min="0"
                    max="100"
                    value={formData.complianceRate}
                    onChange={handleInputChange}
                    placeholder="e.g., 85"
                    required
                  />
                </div>
              </div>
              <div className="grid gap-2">
                <Label htmlFor="notes">Notes (Optional)</Label>
                <Textarea 
                  id="notes"
                  name="notes"
                  value={formData.notes}
                  onChange={handleInputChange}
                  placeholder="Additional information about this department's compliance"
                  rows={3}
                />
              </div>
            </div>
            <DialogFooter>
              <Button 
                type="button" 
                variant="outline" 
                onClick={() => setIsAddDialogOpen(false)}
              >
                Cancel
              </Button>
              <Button 
                type="submit"
                disabled={isSubmitting}
              >
                {isSubmitting ? (
                  <>
                    <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                    Saving...
                  </>
                ) : (
                  'Save Department'
                )}
              </Button>
            </DialogFooter>
          </form>
        </DialogContent>
      </Dialog>
      
      {/* Edit Department Dialog */}
      <Dialog open={isEditDialogOpen} onOpenChange={setIsEditDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Edit Department Compliance</DialogTitle>
          </DialogHeader>
          <form onSubmit={handleSubmit}>
            <div className="grid gap-4 py-4">
              <div className="grid gap-2">
                <Label htmlFor="edit-department">Department Name</Label>
                <Input 
                  id="edit-department"
                  name="department"
                  value={formData.department}
                  onChange={handleInputChange}
                  placeholder="e.g., Nursing"
                  required
                />
              </div>
              <div className="grid grid-cols-2 gap-4">
                <div className="grid gap-2">
                  <Label htmlFor="edit-staffCount">Staff Count</Label>
                  <Input 
                    id="edit-staffCount"
                    name="staffCount"
                    type="number"
                    min="1"
                    value={formData.staffCount}
                    onChange={handleInputChange}
                    placeholder="e.g., 45"
                    required
                  />
                </div>
                <div className="grid gap-2">
                  <Label htmlFor="edit-complianceRate">Compliance Rate (%)</Label>
                  <Input 
                    id="edit-complianceRate"
                    name="complianceRate"
                    type="number"
                    min="0"
                    max="100"
                    value={formData.complianceRate}
                    onChange={handleInputChange}
                    placeholder="e.g., 85"
                    required
                  />
                </div>
              </div>
              <div className="grid gap-2">
                <Label htmlFor="edit-notes">Notes (Optional)</Label>
                <Textarea 
                  id="edit-notes"
                  name="notes"
                  value={formData.notes}
                  onChange={handleInputChange}
                  placeholder="Additional information about this department's compliance"
                  rows={3}
                />
              </div>
            </div>
            <DialogFooter>
              <Button 
                type="button" 
                variant="outline" 
                onClick={() => setIsEditDialogOpen(false)}
              >
                Cancel
              </Button>
              <Button 
                type="submit"
                disabled={isSubmitting}
              >
                {isSubmitting ? (
                  <>
                    <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                    Updating...
                  </>
                ) : (
                  'Update Department'
                )}
              </Button>
            </DialogFooter>
          </form>
        </DialogContent>
      </Dialog>
      
      {/* Delete Confirmation Dialog */}
      <AlertDialog open={isDeleteDialogOpen} onOpenChange={setIsDeleteDialogOpen}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>Are you sure?</AlertDialogTitle>
            <AlertDialogDescription>
              This will permanently delete the compliance record for {currentDept?.name}.
              This action cannot be undone.
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel disabled={isSubmitting}>Cancel</AlertDialogCancel>
            <AlertDialogAction
              onClick={handleDeleteConfirm}
              disabled={isSubmitting}
              className="bg-red-600 hover:bg-red-700"
            >
              {isSubmitting ? (
                <>
                  <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                  Deleting...
                </>
              ) : (
                'Delete'
              )}
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </Card>
  );
}

// Helper function to determine compliance status based on rate
function getComplianceStatus(rate: number): React.ReactNode {
  if (rate >= 90) {
    return <span className="text-green-600">Compliant</span>;
  } else if (rate >= 75) {
    return <span className="text-yellow-600">Nearing Compliance</span>;
  } else {
    return <span className="text-red-600">Non-Compliant</span>;
  }
} 