import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { BarChart, Bar, XAxis, YAxis, CartesianGrid, Tooltip, ResponsiveContainer, Cell } from 'recharts';
import { Info } from "lucide-react";

const data = [
  { name: 'HIPAA', impact: 85, category: 'Privacy' },
  { name: 'JCI', impact: 78, category: 'Quality' },
  { name: 'ISO 9001', impact: 65, category: 'Quality' },
  { name: 'FDA 21 CFR', impact: 60, category: 'Compliance' },
  { name: 'GDPR', impact: 55, category: 'Privacy' },
  { name: 'OSHA', impact: 40, category: 'Safety' },
];

const impactLevels = [
  { name: 'High Impact', color: '#ef4444', range: 'impact >= 70' },
  { name: 'Medium Impact', color: '#f59e0b', range: '50 <= impact < 70' },
  { name: 'Low Impact', color: '#3b82f6', range: 'impact < 50' },
];

const getBarColor = (impact: number) => {
  if (impact >= 70) return '#ef4444';
  if (impact >= 50) return '#f59e0b';
  return '#3b82f6';
};

const impactData = [
  { name: 'Resource Requirements', category: 'Implementation', value: 68 },
  { name: 'Process Changes', category: 'Implementation', value: 72 },
  { name: 'Budget Allocation', category: 'Financial', value: 65 },
  { name: 'Training Needs', category: 'Operational', value: 75 },
  { name: 'Documentation Updates', category: 'Operational', value: 58 },
];

const CustomTooltip = ({ active, payload, label }: any) => {
  if (active && payload && payload.length) {
    return (
      <div className="custom-tooltip bg-white p-3 border border-gray-200 shadow-md rounded-md">
        <p className="font-medium text-sm">{`${label}`}</p>
        <p className="text-sm">{`Impact Score: ${payload[0].value}`}</p>
        <p className="text-xs text-gray-500">{`Category: ${payload[0].payload.category}`}</p>
      </div>
    );
  }
  return null;
};

const RegulatoryImpactAnalysis = () => {
  return (
    <div className="space-y-6">
      <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
        <Card className="md:col-span-2">
          <CardHeader>
            <CardTitle>Regulatory Impact by Standard</CardTitle>
            <CardDescription>Estimated impact of regulatory changes on operations</CardDescription>
          </CardHeader>
          <CardContent>
            <div className="h-[300px]">
              <ResponsiveContainer width="100%" height="100%">
                <BarChart 
                  data={data}
                  margin={{ top: 20, right: 30, left: 20, bottom: 40 }}
                >
                  <CartesianGrid strokeDasharray="3 3" vertical={false} />
                  <XAxis 
                    dataKey="name" 
                    tick={{ fontSize: 12 }}
                    axisLine={{ stroke: '#E5E7EB' }}
                    tickLine={false}
                  />
                  <YAxis 
                    axisLine={{ stroke: '#E5E7EB' }}
                    tickLine={false}
                    tick={{ fontSize: 12 }}
                  />
                  <Tooltip content={<CustomTooltip />} />
                  <Bar dataKey="impact" radius={[4, 4, 0, 0]}>
                    {data.map((entry, index) => (
                      <Cell key={`cell-${index}`} fill={getBarColor(entry.impact)} />
                    ))}
                  </Bar>
                </BarChart>
              </ResponsiveContainer>
            </div>
            <div className="flex justify-center mt-2 gap-6">
              {impactLevels.map((level) => (
                <div key={level.name} className="flex items-center gap-2">
                  <div className="w-3 h-3 rounded-full" style={{ backgroundColor: level.color }}></div>
                  <span className="text-xs text-gray-600">{level.name}</span>
                </div>
              ))}
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle>Impact Categories</CardTitle>
            <CardDescription>Areas affected by regulatory changes</CardDescription>
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              {impactData.map((item, index) => (
                <div key={index} className="space-y-1">
                  <div className="flex justify-between items-center">
                    <span className="text-sm font-medium">{item.name}</span>
                    <Badge className={`${
                      item.value >= 70 ? 'bg-red-100 text-red-800' : 
                      item.value >= 50 ? 'bg-amber-100 text-amber-800' : 
                      'bg-blue-100 text-blue-800'
                    }`}>
                      {item.value}%
                    </Badge>
                  </div>
                  <div className="h-2 bg-gray-100 rounded-full">
                    <div 
                      className="h-full rounded-full" 
                      style={{ 
                        width: `${item.value}%`,
                        backgroundColor: getBarColor(item.value)
                      }}
                    ></div>
                  </div>
                  <div className="text-xs text-gray-500">
                    {item.category}
                  </div>
                </div>
              ))}
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="pt-6">
            <h3 className="text-lg font-semibold mb-4">Recommended Actions</h3>
            <div className="space-y-4">
              <div className="flex items-start space-x-3 p-3 rounded-md bg-blue-50">
                <Info className="h-5 w-5 text-blue-600 mt-0.5" />
                <div>
                  <h4 className="font-medium text-sm text-blue-800">Update Privacy Policies</h4>
                  <p className="text-xs text-blue-700 mt-1">Update all patient privacy policies to align with HIPAA Privacy Updates by next quarter.</p>
                </div>
              </div>
              <div className="flex items-start space-x-3 p-3 rounded-md bg-amber-50">
                <Info className="h-5 w-5 text-amber-600 mt-0.5" />
                <div>
                  <h4 className="font-medium text-sm text-amber-800">JCI Readiness Assessment</h4>
                  <p className="text-xs text-amber-700 mt-1">Conduct a comprehensive gap analysis against new JCI 7th Edition standards.</p>
                </div>
              </div>
              <div className="flex items-start space-x-3 p-3 rounded-md bg-green-50">
                <Info className="h-5 w-5 text-green-600 mt-0.5" />
                <div>
                  <h4 className="font-medium text-sm text-green-800">Staff Training</h4>
                  <p className="text-xs text-green-700 mt-1">Schedule training sessions for all staff on updated medical device reporting requirements.</p>
                </div>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card className="md:col-span-2">
          <CardHeader>
            <CardTitle>Implementation Timeline</CardTitle>
            <CardDescription>Projected timeline for regulatory compliance projects</CardDescription>
          </CardHeader>
          <CardContent>
            <div className="space-y-3">
              <div className="relative">
                <div className="flex items-center">
                  <div className="flex-shrink-0 w-10 h-10 flex items-center justify-center rounded-full bg-blue-100 text-blue-600 font-medium">Q2</div>
                  <div className="ml-4 flex-1">
                    <div className="h-2 bg-blue-500 rounded-full"></div>
                  </div>
                </div>
                <div className="ml-14 mt-1 space-y-2">
                  <div className="text-sm font-medium">Gap Analysis & Planning</div>
                  <div className="flex flex-wrap gap-2">
                    <Badge className="bg-blue-100 text-blue-800">HIPAA Privacy Updates</Badge>
                    <Badge className="bg-blue-100 text-blue-800">Staff Training</Badge>
                  </div>
                </div>
              </div>
              
              <div className="relative">
                <div className="flex items-center">
                  <div className="flex-shrink-0 w-10 h-10 flex items-center justify-center rounded-full bg-amber-100 text-amber-600 font-medium">Q3</div>
                  <div className="ml-4 flex-1">
                    <div className="h-2 bg-amber-500 rounded-full"></div>
                  </div>
                </div>
                <div className="ml-14 mt-1 space-y-2">
                  <div className="text-sm font-medium">Implementation & Training</div>
                  <div className="flex flex-wrap gap-2">
                    <Badge className="bg-amber-100 text-amber-800">JCI Preparation</Badge>
                    <Badge className="bg-amber-100 text-amber-800">ISO 9001 Implementation</Badge>
                  </div>
                </div>
              </div>
              
              <div className="relative">
                <div className="flex items-center">
                  <div className="flex-shrink-0 w-10 h-10 flex items-center justify-center rounded-full bg-green-100 text-green-600 font-medium">Q4</div>
                  <div className="ml-4 flex-1">
                    <div className="h-2 bg-green-500 rounded-full"></div>
                  </div>
                </div>
                <div className="ml-14 mt-1 space-y-2">
                  <div className="text-sm font-medium">Review & Certification</div>
                  <div className="flex flex-wrap gap-2">
                    <Badge className="bg-green-100 text-green-800">JCI Certification</Badge>
                    <Badge className="bg-green-100 text-green-800">Annual Compliance Review</Badge>
                  </div>
                </div>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>
    </div>
  );
};

export default RegulatoryImpactAnalysis; 