import { useState } from 'react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Calendar } from "@/components/ui/calendar";
import { useScheduleData } from '@/hooks/useScheduleData';
import { AlertCircle, Calendar as CalendarIcon, ChevronLeft, ChevronRight } from 'lucide-react';
import { format, addMonths, isSameMonth, isSameDay } from 'date-fns';
import { cn } from '@/lib/utils';

// Convert string dates to Date objects
const convertStringToDate = (dateString: string) => {
  // Parse date strings in format like "June 15, 2023"
  try {
    return new Date(dateString);
  } catch (e) {
    console.error("Error parsing date:", dateString);
    return new Date(); // Fallback to current date
  }
};

const RegulatoryCalendar = () => {
  const { regulatoryDeadlines, auditSchedules } = useScheduleData();
  const [selectedDate, setSelectedDate] = useState<Date | undefined>(new Date());
  const [month, setMonth] = useState<Date>(new Date());
  
  // Combine regulatory deadlines and audit schedules
  const events = [
    ...regulatoryDeadlines.map(deadline => ({
      id: deadline.id,
      title: deadline.title,
      date: convertStringToDate(deadline.dueDate),
      type: 'deadline',
      category: 'regulatory',
      status: deadline.status,
      criticalLevel: deadline.criticalLevel,
      assignedTo: deadline.assignedTo,
      description: deadline.description,
    })),
    ...auditSchedules.map(audit => ({
      id: audit.id,
      title: audit.title,
      date: convertStringToDate(audit.date),
      type: 'audit',
      category: audit.department,
      status: audit.status,
      assignedTo: audit.auditor,
      description: undefined,
    }))
  ];

  // Filter events for the selected date
  const selectedDateEvents = events.filter(event => 
    selectedDate && isSameDay(event.date, selectedDate)
  );

  // Filter events for the current month (for highlighting dates with events)
  const currentMonthEvents = events.filter(event => 
    month && isSameMonth(event.date, month)
  );

  // Create a set of dates that have events (for highlighting in calendar)
  const datesWithEvents = new Set(
    currentMonthEvents.map(event => 
      format(event.date, 'yyyy-MM-dd')
    )
  );

  // Group events by type for the selected date
  const deadlineEvents = selectedDateEvents.filter(event => event.type === 'deadline');
  const auditEvents = selectedDateEvents.filter(event => event.type === 'audit');
  
  const handlePreviousMonth = () => {
    setMonth(prevMonth => addMonths(prevMonth, -1));
  };

  const handleNextMonth = () => {
    setMonth(prevMonth => addMonths(prevMonth, 1));
  };

  // Get today's events
  const isTodaySelected = selectedDate && isSameDay(selectedDate, new Date());
  
  const getEventBorderColor = (event: any) => {
    if (event.type === 'deadline' && 'criticalLevel' in event) {
      switch (event.criticalLevel) {
        case 'high': return 'bg-red-500';
        case 'medium': return 'bg-amber-500';
        default: return 'bg-blue-500';
      }
    }
    return 'bg-blue-500';
  };

  return (
    <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
      <Card className="md:col-span-1 shadow-sm border-gray-200">
        <CardHeader className="pb-2">
          <div className="flex justify-between items-center">
            <CardTitle className="text-lg font-semibold">Regulatory Calendar</CardTitle>
            <div className="flex gap-1">
              <button 
                className="h-8 w-8 p-0 flex items-center justify-center border border-gray-200 rounded hover:bg-gray-50"
                onClick={handlePreviousMonth}
              >
                <ChevronLeft className="h-4 w-4" />
              </button>
              <button 
                className="h-8 w-8 p-0 flex items-center justify-center border border-gray-200 rounded hover:bg-gray-50"
                onClick={handleNextMonth}
              >
                <ChevronRight className="h-4 w-4" />
              </button>
            </div>
          </div>
          <CardDescription className="text-sm text-gray-500">
            {format(month, 'MMMM yyyy')}
          </CardDescription>
        </CardHeader>
        <CardContent>
          <Calendar
            mode="single"
            selected={selectedDate}
            onSelect={setSelectedDate}
            month={month}
            onMonthChange={setMonth}
            className="rounded-md border-0 shadow-none p-0 w-full"
            modifiers={{
              eventDay: (date) => 
                datesWithEvents.has(format(date, 'yyyy-MM-dd'))
            }}
            modifiersClassNames={{
              eventDay: "font-bold text-blue-600"
            }}
          />
          <div className="mt-4 flex flex-col gap-2">
            <div className="text-sm font-medium text-gray-700">Legend</div>
            <div className="flex flex-col gap-1.5">
              <div className="flex items-center gap-2 text-xs">
                <span className="inline-block w-3 h-3 bg-red-500 rounded-full"></span>
                <span>High priority deadlines</span>
              </div>
              <div className="flex items-center gap-2 text-xs">
                <span className="inline-block w-3 h-3 bg-amber-500 rounded-full"></span>
                <span>Medium priority deadlines</span>
              </div>
              <div className="flex items-center gap-2 text-xs">
                <span className="inline-block w-3 h-3 bg-blue-500 rounded-full"></span>
                <span>Standard events & audits</span>
              </div>
            </div>
          </div>
        </CardContent>
      </Card>
      
      <Card className="md:col-span-2 shadow-sm border-gray-200">
        <CardHeader className="pb-3 border-b border-gray-100">
          <div className="flex justify-between items-center">
            <div>
              <CardTitle className="text-lg font-semibold flex items-center gap-2">
                <CalendarIcon className="h-5 w-5 text-blue-600" />
                <span>{selectedDate ? format(selectedDate, 'EEEE, MMMM d, yyyy') : 'Selected Date'}</span>
                {isTodaySelected && <Badge className="bg-blue-100 text-blue-800">Today</Badge>}
              </CardTitle>
              <CardDescription className="mt-1">
                {selectedDateEvents.length 
                  ? `${selectedDateEvents.length} event${selectedDateEvents.length !== 1 ? 's' : ''}` 
                  : 'No events scheduled for this date'}
              </CardDescription>
            </div>
          </div>
        </CardHeader>
        <CardContent className="p-0">
          <div className="h-[350px] overflow-auto">
            {selectedDateEvents.length > 0 ? (
              <div className="divide-y divide-gray-100">
                {deadlineEvents.length > 0 && (
                  <div className="p-4">
                    <h3 className="text-sm font-medium text-gray-500 mb-3">Regulatory Deadlines</h3>
                    <div className="space-y-3">
                      {deadlineEvents.map((event) => (
                        <Card key={event.id} className="overflow-hidden border-gray-200 shadow-none hover:bg-gray-50 transition-colors">
                          <div className="flex">
                            <div className={`w-1 h-full ${getEventBorderColor(event)}`}></div>
                            <div className="flex-1 p-3">
                              <div className="flex items-start justify-between">
                                <div className="space-y-1">
                                  <h3 className="font-medium text-sm">{event.title}</h3>
                                  <div className="flex items-center gap-2">
                                    <Badge className="bg-amber-100 text-amber-800">
                                      Regulatory
                                    </Badge>
                                    {'criticalLevel' in event && (
                                      <Badge className={cn(
                                        event.criticalLevel === 'high' ? 'bg-red-100 text-red-800' : 
                                        event.criticalLevel === 'medium' ? 'bg-amber-100 text-amber-800' : 
                                        'bg-blue-100 text-blue-800'
                                      )}>
                                        {event.criticalLevel === 'high' ? 'High Priority' : 
                                         event.criticalLevel === 'medium' ? 'Medium Priority' : 'Low Priority'}
                                      </Badge>
                                    )}
                                  </div>
                                </div>
                                <Badge variant="outline" className="capitalize">
                                  {event.status}
                                </Badge>
                              </div>
                              <div className="mt-2 text-xs text-gray-500">
                                <p>Assigned to: {event.assignedTo}</p>
                                {event.description && <p className="mt-1">{event.description}</p>}
                              </div>
                            </div>
                          </div>
                        </Card>
                      ))}
                    </div>
                  </div>
                )}
                
                {auditEvents.length > 0 && (
                  <div className="p-4">
                    <h3 className="text-sm font-medium text-gray-500 mb-3">Audits</h3>
                    <div className="space-y-3">
                      {auditEvents.map((event) => (
                        <Card key={event.id} className="overflow-hidden border-gray-200 shadow-none hover:bg-gray-50 transition-colors">
                          <div className="flex">
                            <div className="w-1 h-full bg-blue-500"></div>
                            <div className="flex-1 p-3">
                              <div className="flex items-start justify-between">
                                <div className="space-y-1">
                                  <h3 className="font-medium text-sm">{event.title}</h3>
                                  <div className="flex items-center gap-2">
                                    <Badge className="bg-blue-100 text-blue-800">
                                      Audit
                                    </Badge>
                                    <span className="text-xs text-gray-500">{event.category}</span>
                                  </div>
                                </div>
                                <Badge variant="outline" className="capitalize">
                                  {event.status}
                                </Badge>
                              </div>
                              <div className="mt-2 text-xs text-gray-500">
                                <p>Auditor: {event.assignedTo}</p>
                              </div>
                            </div>
                          </div>
                        </Card>
                      ))}
                    </div>
                  </div>
                )}
              </div>
            ) : (
              <div className="flex flex-col items-center justify-center h-full text-center p-6">
                <AlertCircle className="h-10 w-10 text-gray-300 mb-2" />
                <h3 className="font-medium text-gray-500">No events scheduled</h3>
                <p className="text-sm text-gray-400 mt-1 max-w-xs">
                  There are no regulatory deadlines or audits scheduled for this date. Select a different date to view events.
                </p>
              </div>
            )}
          </div>
        </CardContent>
      </Card>
    </div>
  );
};

export default RegulatoryCalendar; 