'use client'

import { useState, useEffect, useMemo, useCallback } from 'react'
import Link from 'next/link'
import { usePathname, useRouter } from 'next/navigation'
import { signOut, useSession } from 'next-auth/react'
import { 
  DropdownMenu, 
  DropdownMenuContent, 
  DropdownMenuItem, 
  DropdownMenuLabel, 
  DropdownMenuSeparator, 
  DropdownMenuTrigger 
} from '@/components/ui/dropdown-menu'
import { UserIcon, SettingsIcon as SettingsIconDropdown, LogOutIcon, BuildingIcon } from '@/components/ui/icons'
import { Avatar, AvatarFallback, AvatarImage } from '@/components/ui/avatar'
import SuperadminNavigation from './SuperadminNavigation'
import SidebarRecentStandards from './SidebarRecentStandards'

// Icons - Memoized to prevent re-creation
const ChevronLeft = () => (
  <svg
    xmlns="http://www.w3.org/2000/svg"
    width="24"
    height="24"
    viewBox="0 0 24 24"
    fill="none"
    stroke="currentColor"
    strokeWidth="2"
    strokeLinecap="round"
    strokeLinejoin="round"
    className="h-4 w-4"
  >
    <path d="m15 18-6-6 6-6" />
  </svg>
)

const ChevronRight = () => (
  <svg
    xmlns="http://www.w3.org/2000/svg"
    width="24"
    height="24"
    viewBox="0 0 24 24"
    fill="none"
    stroke="currentColor"
    strokeWidth="2"
    strokeLinecap="round"
    strokeLinejoin="round"
    className="h-4 w-4"
  >
    <path d="m9 18 6-6-6-6" />
  </svg>
)

const Hospital = () => (
  <svg
    xmlns="http://www.w3.org/2000/svg"
    width="24"
    height="24"
    viewBox="0 0 24 24"
    fill="none"
    stroke="currentColor"
    strokeWidth="2"
    strokeLinecap="round"
    strokeLinejoin="round"
    className="h-6 w-6 text-blue-600"
  >
    <path d="M8 3v2" />
    <path d="M16 3v2" />
    <path d="M21 12h-3a1 1 0 0 1-1-1V7a1 1 0 0 0-1-1H8a1 1 0 0 0-1 1v4a1 1 0 0 1-1 1H3" />
    <path d="M7 14a1 1 0 0 1 1-1h8a1 1 0 0 1 1 1v7H7z" />
    <path d="M12 3v16" />
  </svg>
)

// Helper function to merge classes
const cn = (...classes: (string | undefined)[]) => {
  return classes.filter(Boolean).join(' ')
}

// Optimized Button component with memoization
const Button = ({ 
  className = '', 
  children, 
  onClick,
  variant = 'default',
  size = 'default',
  ...props 
}: React.ButtonHTMLAttributes<HTMLButtonElement> & {
  variant?: 'default' | 'ghost'
  size?: 'default' | 'icon'
}) => {
  const variantClasses = useMemo(() => ({
    default: 'bg-blue-600 text-white hover:bg-blue-700',
    ghost: 'hover:bg-gray-100'
  }), [])
  
  const sizeClasses = useMemo(() => ({
    default: 'px-4 py-2',
    icon: 'w-9 h-9 p-0'
  }), [])
  
  return (
    <button
      className={cn(
        "inline-flex items-center justify-center rounded-md text-sm font-medium transition-colors focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-blue-500",
        variantClasses[variant],
        sizeClasses[size],
        className
      )}
      onClick={onClick}
      {...props}
    >
      {children}
    </button>
  )
}

// Separator component
const Separator = ({ className }: { className?: string }) => (
  <div className={cn("h-[1px] w-full bg-gray-200", className)} />
)

// Enhanced NavigationLink component with prefetching and optimization
const NavigationLink = ({ 
  item, 
  pathname, 
  collapsed 
}: { 
  item: any, 
  pathname: string, 
  collapsed: boolean 
}) => {
  const isActive = pathname === item.path
  
  return (
    <Link
      href={item.path}
      prefetch={true} // Enable prefetching for faster navigation
      className={cn(
        "flex items-center rounded-md px-3 py-2 text-sm font-medium transition-all duration-150", // Faster transition
        isActive 
          ? "bg-blue-100 text-blue-700" 
          : "text-gray-700 hover:bg-gray-100",
        collapsed ? "justify-center" : ""
      )}
    >
      <item.icon />
      {!collapsed && <span className="ml-3">{item.title}</span>}
      {item.alert && !collapsed && (
        <span className="h-2 w-2 ml-auto rounded-full bg-red-500"></span>
      )}
      {item.alert && collapsed && (
        <span className="h-2 w-2 absolute top-1 right-1 rounded-full bg-red-500"></span>
      )}
    </Link>
  )
}

interface SidebarProps {
  collapsed?: boolean
  onToggle?: () => void
}

const Sidebar = ({ collapsed: propCollapsed, onToggle }: SidebarProps) => {
  const [collapsed, setCollapsed] = useState(propCollapsed || false)
  const pathname = usePathname()
  const router = useRouter()
  const { data: session } = useSession()
  
  // Update internal state when prop changes
  useEffect(() => {
    if (propCollapsed !== undefined) {
      setCollapsed(propCollapsed)
    }
  }, [propCollapsed])

  // Memoized toggle function
  const toggleCollapsed = useCallback(() => {
    setCollapsed(!collapsed)
    if (onToggle) onToggle()
  }, [collapsed, onToggle])

  // Memoized user role checks
  const userRoles = useMemo(() => ({
    isAdmin: session?.user?.role === 'admin' || session?.user?.role === 'super_admin',
    isSuperAdmin: session?.user?.role === 'super_admin'
  }), [session?.user?.role])

  // Memoized navigation items to prevent re-creation on each render
  const navigationItems = useMemo(() => [
    { title: 'Dashboard', path: '/dashboard', icon: DashboardIcon },
    { title: 'Compliance', path: '/compliance', icon: ComplianceIcon, alert: true },
    { title: 'Findings', path: '/findings', icon: FindingsIcon },
    { title: 'Reports', path: '/reports', icon: ReportsIcon },
    { title: 'Documents', path: '/documents', icon: DocumentsIcon },
    { title: 'Regulations', path: '/regulations', icon: RegulationsIcon },
    { title: 'Schedule', path: '/schedule', icon: ScheduleIcon, alert: true },
    { title: 'Staff & Training', path: '/training', icon: StaffIcon },
    { title: 'Departments', path: '/hospital/departments', icon: BuildingIcon },
  ], [])

  // Memoized admin navigation items
  const adminItems = useMemo(() => [
    { title: 'Hospitals', path: '/admin/hospitals', icon: HospitalIcon },
    { title: 'All Departments', path: '/admin/departments', icon: BuildingIcon },
    { title: 'Settings', path: '/admin/settings', icon: SettingsIconDropdown },
    { title: 'Security', path: '/admin/security', icon: SecurityIcon, badge: 'New' },
    { title: 'Database', path: '/admin/database', icon: DatabaseIcon },
    { title: 'Access', path: '/admin/access', icon: AccessIcon },
  ], [])

  // Memoized navigation list to prevent unnecessary re-renders
  const navigationList = useMemo(() => 
    navigationItems.map((item) => (
      <NavigationLink 
        key={item.path}
        item={item}
        pathname={pathname}
        collapsed={collapsed}
      />
    )), [navigationItems, pathname, collapsed])
  
  return (
    <div 
      className={cn(
        "h-screen flex flex-col border-r border-gray-200 bg-white transition-all duration-300",
        collapsed ? "w-16" : "w-64"
      )}
    >
      <div className="flex items-center justify-between p-4">
        {!collapsed && (
          <div className="flex items-center gap-2">
            <Hospital />
            <span className="font-bold text-lg text-blue-600">Health Guardian</span>
          </div>
        )}
        {collapsed && (
          <Hospital />
        )}
        <Button
          variant="ghost"
          size="icon"
          onClick={toggleCollapsed}
          className="ml-auto"
          aria-label={collapsed ? "Expand sidebar" : "Collapse sidebar"}
        >
          {collapsed ? <ChevronRight /> : <ChevronLeft />}
        </Button>
      </div>
      
      <Separator />
      
      <div className="flex-1 overflow-y-auto py-4 px-3">
        <nav className="space-y-1">
          {navigationList}
        </nav>
        
        {/* Replace static standards section with dynamic component */}
        <SidebarRecentStandards collapsed={collapsed} />
        
        {/* SuperAdmin Section - only for superadmin users */}
        <SuperadminNavigation collapsed={collapsed} />
      </div>
      
      <div className="p-4 border-t border-gray-200">
        <DropdownMenu>
          <DropdownMenuTrigger asChild>
            <div className={cn(
              "flex items-center cursor-pointer",
              collapsed ? "justify-center" : "space-x-3"
            )}>
              <Avatar>
                <AvatarImage src="https://ui-avatars.com/api/?name=Admin+User&background=0D8ABC&color=fff" />
                <AvatarFallback style={{ backgroundColor: '#3b82f6', color: 'white' }}>
                  {session?.user?.name ? session.user.name.charAt(0) : 'U'}
                </AvatarFallback>
              </Avatar>
              {!collapsed && (
                <div className="flex-1 min-w-0">
                  <p className="text-sm font-medium text-gray-900 truncate">
                    {session?.user?.name || 'User'}
                  </p>
                  <p className="text-xs text-gray-500 truncate">
                    {session?.user?.role || 'Role'}
                  </p>
                </div>
              )}
            </div>
          </DropdownMenuTrigger>
          <DropdownMenuContent className="w-56" align="end" forceMount>
            <DropdownMenuLabel className="font-normal">
              <div className="flex flex-col space-y-1">
                <p className="text-sm font-medium leading-none">
                  {session?.user?.name || 'User'}
                </p>
                <p className="text-xs leading-none text-muted-foreground">
                  {session?.user?.email}
                </p>
              </div>
            </DropdownMenuLabel>
            <DropdownMenuSeparator />
            <DropdownMenuItem>
              <UserIcon className="mr-2 h-4 w-4" />
              <span>Profile</span>
            </DropdownMenuItem>
            <DropdownMenuItem>
              <SettingsIconDropdown className="mr-2 h-4 w-4" />
              <span>Settings</span>
            </DropdownMenuItem>
            <DropdownMenuSeparator />
            <DropdownMenuItem onClick={() => signOut()}>
              <LogOutIcon className="mr-2 h-4 w-4" />
              <span>Log out</span>
            </DropdownMenuItem>
          </DropdownMenuContent>
        </DropdownMenu>
      </div>
    </div>
  )
}

// Icon Components
const DashboardIcon = () => (
  <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="w-5 h-5">
    <rect width="7" height="9" x="3" y="3" rx="1" />
    <rect width="7" height="5" x="14" y="3" rx="1" />
    <rect width="7" height="9" x="14" y="12" rx="1" />
    <rect width="7" height="5" x="3" y="16" rx="1" />
  </svg>
)

const ComplianceIcon = () => (
  <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="w-5 h-5">
    <path d="M12 22c5.523 0 10-4.477 10-10S17.523 2 12 2 2 6.477 2 12s4.477 10 10 10z" />
    <path d="m9 12 2 2 4-4" />
  </svg>
)

const FindingsIcon = () => (
  <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="w-5 h-5">
    <path d="M14.5 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V7.5L14.5 2z" />
    <polyline points="14 2 14 8 20 8" />
    <line x1="16" x2="8" y1="13" y2="13" />
    <line x1="16" x2="8" y1="17" y2="17" />
    <line x1="10" x2="8" y1="9" y2="9" />
  </svg>
)

const ReportsIcon = () => (
  <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="w-5 h-5">
    <path d="M6 9H4.5a2.5 2.5 0 0 0 0 5H6" />
    <path d="M18 9h1.5a2.5 2.5 0 0 1 0 5H18" />
    <path d="M8 9v7" />
    <path d="M16 9v7" />
    <path d="M12 9v8" />
    <path d="M12 17a2 2 0 0 1-2 2H6.5a2.5 2.5 0 0 1 0-5H10" />
    <path d="M12 17a2 2 0 0 0 2 2h3.5a2.5 2.5 0 0 0 0-5H14" />
  </svg>
)

const DocumentsIcon = () => (
  <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="w-5 h-5">
    <path d="M20 7h-3a2 2 0 0 1-2-2V2" />
    <path d="M9 18a2 2 0 0 1-2-2V4a2 2 0 0 1 2-2h7l4 4v10a2 2 0 0 1-2 2H9Z" />
    <path d="M3 7.6v12.8A1.6 1.6 0 0 0 4.6 22h9.8" />
  </svg>
)

const RegulationsIcon = () => (
  <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="w-5 h-5">
    <path d="M4 19.5v-15A2.5 2.5 0 0 1 6.5 2H20v20H6.5a2.5 2.5 0 0 1 0-5H20" />
  </svg>
)

const ScheduleIcon = () => (
  <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="w-5 h-5">
    <rect width="18" height="18" x="3" y="4" rx="2" ry="2" />
    <line x1="16" x2="16" y1="2" y2="6" />
    <line x1="8" x2="8" y1="2" y2="6" />
    <line x1="3" x2="21" y1="10" y2="10" />
    <path d="m9 16 2 2 4-4" />
  </svg>
)

const StaffIcon = () => (
  <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="w-5 h-5">
    <path d="M16 21v-2a4 4 0 0 0-4-4H6a4 4 0 0 0-4 4v2" />
    <circle cx="9" cy="7" r="4" />
    <path d="M22 21v-2a4 4 0 0 0-3-3.87" />
    <path d="M16 3.13a4 4 0 0 1 0 7.75" />
  </svg>
)

const HospitalIcon = () => (
  <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="w-5 h-5">
    <path d="M8 3v2" />
    <path d="M16 3v2" />
    <path d="M21 12h-3a1 1 0 0 1-1-1V7a1 1 0 0 0-1-1H8a1 1 0 0 0-1 1v4a1 1 0 0 1-1 1H3" />
    <path d="M7 14a1 1 0 0 1 1-1h8a1 1 0 0 1 1 1v7H7z" />
    <path d="M12 3v16" />
  </svg>
)

const UsersIcon = () => (
  <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="w-5 h-5">
    <path d="M16 21v-2a4 4 0 0 0-4-4H6a4 4 0 0 0-4 4v2" />
    <circle cx="9" cy="7" r="4" />
    <path d="M22 21v-2a4 4 0 0 0-3-3.87" />
    <path d="M16 3.13a4 4 0 0 1 0 7.75" />
  </svg>
)

const SettingsIcon = () => (
  <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="w-5 h-5">
    <path d="M12.22 2h-.44a2 2 0 0 0-2 2v.18a2 2 0 0 1-1 1.73l-.43.25a2 2 0 0 1-2 0l-.15-.08a2 2 0 0 0-2.73.73l-.22.38a2 2 0 0 0 .73 2.73l.15.1a2 2 0 0 1 1 1.72v.51a2 2 0 0 1-1 1.74l-.15.09a2 2 0 0 0-.73 2.73l.22.38a2 2 0 0 0 2.73.73l.15-.08a2 2 0 0 1 2 0l.43.25a2 2 0 0 1 1 1.73V20a2 2 0 0 0 2 2h.44a2 2 0 0 0 2-2v-.18a2 2 0 0 1 1-1.73l.43-.25a2 2 0 0 1 2 0l.15.08a2 2 0 0 0 2.73-.73l.22-.39a2 2 0 0 0-.73-2.73l-.15-.08a2 2 0 0 1-1-1.74v-.5a2 2 0 0 1 1-1.74l.15-.09a2 2 0 0 0 .73-2.73l-.22-.38a2 2 0 0 0-2.73-.73l-.15.08a2 2 0 0 1-2 0l-.43-.25a2 2 0 0 1-1-1.73V4a2 2 0 0 0-2-2z" />
    <circle cx="12" cy="12" r="3" />
  </svg>
)

const SecurityIcon = () => (
  <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="w-5 h-5">
    <circle cx="12" cy="12" r="10" />
    <path d="M12 2a4.5 4.5 0 0 0 0 9 4.5 4.5 0 0 1 0 9 10 10 0 0 0 0-18Z" />
    <path d="M12 2c-1.4 0-2.8.5-4 1.5" />
  </svg>
)

const DatabaseIcon = () => (
  <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="w-5 h-5">
    <ellipse cx="12" cy="5" rx="9" ry="3" />
    <path d="M21 12c0 1.66-4 3-9 3s-9-1.34-9-3" />
    <path d="M3 5v14c0 1.66 4 3 9 3s9-1.34 9-3V5" />
  </svg>
)

const AccessIcon = () => (
  <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="w-5 h-5">
    <rect width="18" height="11" x="3" y="11" rx="2" ry="2" />
    <path d="M7 11V7a5 5 0 0 1 10 0v4" />
    <circle cx="12" cy="16" r="1" />
  </svg>
)

export default Sidebar 