'use client'

import React, { useState } from 'react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { 
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuLabel,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from '@/components/ui/dropdown-menu';
import { ScrollArea } from '@/components/ui/scroll-area';
import { BuildingIcon, SearchIcon, CheckIcon, LoaderIcon } from '@/components/ui/icons';
import { useHospital } from '@/lib/context/HospitalContext';
import { toast } from 'sonner';

const HospitalSelector = () => {
  const { currentHospital, accessibleHospitals, setCurrentHospital, loading, error, refetchHospitals } = useHospital();
  const [searchQuery, setSearchQuery] = useState('');
  const [isDropdownOpen, setIsDropdownOpen] = useState(false);
  
  // Handle the case when currentHospital is null or we're still loading
  if (!currentHospital) {
    return (
      <Button variant="outline" className="flex items-center gap-2 min-w-[180px] justify-start" disabled>
        <BuildingIcon className="h-4 w-4" />
        {loading ? (
          <span className="flex items-center gap-2">
            <LoaderIcon className="h-3 w-3 animate-spin" /> Loading...
          </span>
        ) : (
          <span>No hospital selected</span>
        )}
      </Button>
    );
  }
  
  const filteredHospitals = accessibleHospitals.filter(hospital =>
    hospital.name.toLowerCase().includes(searchQuery.toLowerCase()) ||
    hospital.location.toLowerCase().includes(searchQuery.toLowerCase())
  );
  
  const handleHospitalChange = (hospital: typeof currentHospital) => {
    // Don't do anything if the same hospital is selected
    if (hospital.id === currentHospital.id) {
      setIsDropdownOpen(false);
      return;
    }

    // Close the dropdown and reset search
    setIsDropdownOpen(false);
    setSearchQuery('');
    
    // Change the hospital (context will handle refresh)
    setCurrentHospital(hospital);
  };

  return (
    <DropdownMenu open={isDropdownOpen} onOpenChange={setIsDropdownOpen}>
      <DropdownMenuTrigger asChild>
        <Button 
          variant="outline" 
          className={`flex items-center gap-2 min-w-[180px] justify-start ${currentHospital ? 'border-primary' : ''}`}
        >
          <BuildingIcon className={`h-4 w-4 ${currentHospital ? 'text-primary' : ''}`} />
          <span className="max-w-[150px] truncate">
            {currentHospital ? currentHospital.name : 'Select Hospital'}
          </span>
          <span className="ml-1 flex h-2 w-2 rounded-full bg-primary"></span>
        </Button>
      </DropdownMenuTrigger>
      <DropdownMenuContent align="end" className="w-80">
        <div className="p-2">
          <div className="relative">
            <SearchIcon className="absolute left-2 top-2.5 h-4 w-4 text-muted-foreground" />
            <Input 
              placeholder="Search hospitals..." 
              className="pl-8"
              value={searchQuery}
              onChange={(e) => setSearchQuery(e.target.value)}
            />
          </div>
        </div>
        
        <DropdownMenuSeparator />
        
        <div className="flex items-center justify-between px-2">
          <DropdownMenuLabel>
            All Hospitals
          </DropdownMenuLabel>
          
          {loading ? (
            <div className="flex items-center text-xs text-muted-foreground gap-1">
              <LoaderIcon className="h-3 w-3 animate-spin" />
              Loading...
            </div>
          ) : error ? (
            <Button 
              variant="ghost" 
              size="sm" 
              className="h-8 text-xs" 
              onClick={() => refetchHospitals()}
            >
              Retry
            </Button>
          ) : null}
        </div>
        
        <ScrollArea className="h-[300px]">
          {loading ? (
            <div className="py-6 flex justify-center">
              <LoaderIcon className="h-5 w-5 animate-spin text-muted-foreground" />
            </div>
          ) : error ? (
            <div className="py-4 px-3 text-sm text-center text-muted-foreground">
              <p className="text-red-500 mb-2">{error}</p>
              <Button 
                variant="outline" 
                size="sm" 
                onClick={() => refetchHospitals()}
              >
                Try Again
              </Button>
            </div>
          ) : filteredHospitals.length === 0 ? (
            <div className="py-2 px-2 text-sm text-center text-muted-foreground">
              No hospitals found
            </div>
          ) : (
            filteredHospitals.map((hospital) => (
              <DropdownMenuItem
                key={hospital.id}
                onClick={() => handleHospitalChange(hospital)}
                className="cursor-pointer p-2 flex items-start focus:bg-accent"
              >
                <div className="flex flex-1 items-start gap-3">
                  <div className="bg-muted rounded-md p-1">
                    <BuildingIcon className="h-4 w-4" />
                  </div>
                  <div className="flex-1">
                    <p className="text-sm font-medium">{hospital.name}</p>
                    <p className="text-xs text-muted-foreground">{hospital.location}</p>
                  </div>
                  {currentHospital.id === hospital.id && (
                    <CheckIcon className="h-4 w-4 text-primary" />
                  )}
                </div>
              </DropdownMenuItem>
            ))
          )}
        </ScrollArea>
      </DropdownMenuContent>
    </DropdownMenu>
  );
};

export default HospitalSelector; 