import { useState, useEffect } from 'react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Separator } from '@/components/ui/separator';
import { Button } from '@/components/ui/button';
import { BookOpen, CheckCircle2, ListChecks } from 'lucide-react';
import { Badge } from '@/components/ui/badge';
import Link from 'next/link';
import { format } from 'date-fns';

// Import the StandardItem interface
import { StandardItem } from '@/hooks/useComplianceData';

export const RecentStandards = () => {
  const [standards, setStandards] = useState<StandardItem[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    const fetchStandards = async () => {
      try {
        const response = await fetch('/api/compliance/standards');
        
        if (!response.ok) {
          throw new Error('Failed to fetch standards data');
        }
        
        const data = await response.json();
        
        // Sort by created_at date in descending order and take the 3 most recent
        const sortedStandards = [...data].sort((a, b) => 
          new Date(b.created_at).getTime() - new Date(a.created_at).getTime()
        ).slice(0, 3);
        
        setStandards(sortedStandards);
      } catch (err) {
        console.error('Error fetching recent standards:', err);
        setError((err as Error).message);
      } finally {
        setLoading(false);
      }
    };

    fetchStandards();
  }, []);

  const getComplianceClass = (compliance: number, requiredThreshold: number) => {
    if (compliance >= requiredThreshold) return "bg-green-100 text-green-800";
    if (compliance >= requiredThreshold * 0.8) return "bg-yellow-100 text-yellow-800";
    return "bg-red-100 text-red-800";
  };

  return (
    <Card className="h-full">
      <CardHeader className="pb-2">
        <div className="flex items-center space-x-2">
          <ListChecks className="h-5 w-5 text-blue-600" />
          <CardTitle>Recent Standards</CardTitle>
        </div>
        <CardDescription>Latest regulatory standards added</CardDescription>
      </CardHeader>
      <CardContent className="pt-0">
        {loading ? (
          <div className="space-y-4">
            {[1, 2, 3].map(i => (
              <div key={i} className="animate-pulse">
                <div className="h-4 bg-gray-200 rounded w-3/4 mb-2"></div>
                <div className="h-3 bg-gray-200 rounded w-1/2 mb-1"></div>
                <div className="flex justify-between mt-2">
                  <div className="h-3 bg-gray-200 rounded w-1/4"></div>
                  <div className="h-3 bg-gray-200 rounded w-1/4"></div>
                </div>
                {i < 3 && <Separator className="my-4" />}
              </div>
            ))}
          </div>
        ) : error ? (
          <div className="text-red-500 text-sm">{error}</div>
        ) : standards.length === 0 ? (
          <div className="text-gray-500 text-sm py-4 text-center">
            No standards available yet
          </div>
        ) : (
          <div className="space-y-4">
            {standards.map((standard, index) => (
              <div key={standard.id}>
                {index > 0 && <Separator className="my-4" />}
                <div>
                  <h4 className="font-medium text-sm">{standard.standard}</h4>
                  <div className="flex items-center mt-1">
                    <span className="text-xs text-gray-500">{standard.category}</span>
                    <span className="mx-1.5 text-gray-300">•</span>
                    <span className="text-xs text-gray-500">
                      {format(new Date(standard.created_at), 'MMM d, yyyy')}
                    </span>
                  </div>
                  <div className="mt-2 flex items-center justify-between">
                    <Badge className={getComplianceClass(standard.compliance, standard.required_threshold)}>
                      {standard.compliance}% Compliant
                    </Badge>
                    <Link
                      href="/compliance"
                      className="text-xs text-blue-600 hover:underline flex items-center"
                    >
                      <CheckCircle2 className="h-3 w-3 mr-1" />
                      View details
                    </Link>
                  </div>
                </div>
              </div>
            ))}
          </div>
        )}
        <div className="mt-4 pt-4 border-t text-center">
          <Link href="/compliance">
            <Button variant="outline" size="sm" className="text-xs w-full">
              View All Standards
            </Button>
          </Link>
        </div>
      </CardContent>
    </Card>
  );
};

export default RecentStandards; 