import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { useReportsData, useReportTypes, addHospitalIdHeader } from "@/hooks/useComplianceData";
import { Skeleton } from "@/components/ui/skeleton";
import { format } from "date-fns";
import { toast } from "sonner";
import { useState } from "react";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { useSession } from "next-auth/react";

const ReportsTabContent = () => {
  const { data: session } = useSession();
  const { reports, loading, error, refetch } = useReportsData();
  const { reportTypes, loading: reportTypesLoading } = useReportTypes();
  const [selectedReportType, setSelectedReportType] = useState("");
  const [reportTitle, setReportTitle] = useState("");
  const [submitting, setSubmitting] = useState(false);
  
  const handleGenerateReport = async () => {
    if (!selectedReportType) {
      toast.error("Please select a report type");
      return;
    }
    
    if (!reportTitle) {
      toast.error("Please enter a report title");
      return;
    }
    
    setSubmitting(true);
    
    try {
      const isSuperAdmin = session?.user?.role === 'super_admin';
      const headers = addHospitalIdHeader({ 
        'Content-Type': 'application/json',
      }, Boolean(isSuperAdmin));
      
      const response = await fetch("/api/compliance/reports", {
        method: "POST",
        headers,
        body: JSON.stringify({
          title: reportTitle,
          type: selectedReportType,
        }),
      });
      
      if (!response.ok) {
        const errorData = await response.json();
        console.error("API Error:", errorData);
        throw new Error(errorData.error || errorData.message || "Failed to create report");
      }
      
      toast.success("Report generated successfully");
      setReportTitle("");
      setSelectedReportType("");
      refetch();
    } catch (err) {
      console.error("Error generating report:", err);
      toast.error((err as Error).message);
    } finally {
      setSubmitting(false);
    }
  };
  
  if (loading || reportTypesLoading) {
    return <ReportsTabSkeleton />;
  }
  
  if (error) {
    return <div className="p-4 text-red-500">Error loading reports data: {error}</div>;
  }

  return (
    <Card>
      <CardHeader>
        <CardTitle>Compliance Reports</CardTitle>
        <CardDescription>Generate and view compliance reports</CardDescription>
      </CardHeader>
      <CardContent>
        <div className="space-y-4">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div className="border rounded-md p-4">
              <h3 className="font-medium mb-2">Recent Compliance Reports</h3>
              {reports.length > 0 ? (
                <ul className="space-y-2">
                  {reports.map(report => (
                    <li key={report.id} className="text-sm">
                      <div className="flex justify-between items-center">
                        <span>{report.title}</span>
                        <span className="text-xs text-gray-500">
                          {format(new Date(report.created_at), 'MMM dd, yyyy')}
                        </span>
                      </div>
                    </li>
                  ))}
                </ul>
              ) : (
                <p className="text-sm text-gray-500">No reports available</p>
              )}
            </div>
            <div className="border rounded-md p-4">
              <h3 className="font-medium mb-2">Generate New Report</h3>
              <div className="space-y-3">
                <div>
                  <label htmlFor="reportTitle" className="block text-sm font-medium mb-1">Report Title</label>
                  <Input
                    id="reportTitle"
                    placeholder="Enter report title"
                    value={reportTitle}
                    onChange={(e) => setReportTitle(e.target.value)}
                    className="w-full mb-2"
                  />
                </div>
                <div>
                  <label htmlFor="reportType" className="block text-sm font-medium mb-1">Report Type</label>
                  <Select value={selectedReportType} onValueChange={setSelectedReportType}>
                    <SelectTrigger className="w-full mb-2">
                      <SelectValue placeholder="Select Report Type" />
                    </SelectTrigger>
                    <SelectContent>
                      {reportTypes.map((type: {id: number, value: string, label: string}) => (
                        <SelectItem key={type.value} value={type.value}>
                          {type.label}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>
                <Button 
                  className="w-full bg-blue-600 text-white px-4 py-2 rounded hover:bg-blue-700"
                  onClick={handleGenerateReport}
                  disabled={submitting}
                >
                  {submitting ? "Generating..." : "Generate Report"}
                </Button>
              </div>
            </div>
          </div>
        </div>
      </CardContent>
    </Card>
  );
};

const ReportsTabSkeleton = () => {
  return (
    <Card>
      <CardHeader>
        <CardTitle>Compliance Reports</CardTitle>
        <CardDescription>Generate and view compliance reports</CardDescription>
      </CardHeader>
      <CardContent>
        <div className="space-y-4">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div className="border rounded-md p-4">
              <h3 className="font-medium mb-2">Recent Compliance Reports</h3>
              <ul className="space-y-2">
                {[1, 2, 3].map(i => (
                  <li key={i} className="text-sm">
                    <Skeleton className="h-5 w-full" />
                  </li>
                ))}
              </ul>
            </div>
            <div className="border rounded-md p-4">
              <h3 className="font-medium mb-2">Generate New Report</h3>
              <Skeleton className="h-10 w-full mb-2" />
              <Skeleton className="h-10 w-full" />
            </div>
          </div>
        </div>
      </CardContent>
    </Card>
  );
};

export default ReportsTabContent; 