import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { usePoliciesData, usePolicyStatuses, usePolicyCategories, addHospitalIdHeader } from "@/hooks/useComplianceData";
import { Skeleton } from "@/components/ui/skeleton";
import { toast } from "sonner";
import { useState } from "react";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Plus, X } from "lucide-react";
import { useSession } from "next-auth/react";

const PoliciesTab = () => {
  const { data: session } = useSession();
  const { policies, loading, error, refetch } = usePoliciesData();
  const { statuses: policyStatuses, loading: statusesLoading } = usePolicyStatuses();
  const { categories: policyCategories, loading: categoriesLoading } = usePolicyCategories();
  const [showForm, setShowForm] = useState(false);
  const [formData, setFormData] = useState({
    name: "",
    category: "",
    status: ""
  });
  const [submitting, setSubmitting] = useState(false);

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const { name, value } = e.target;
    setFormData((prev) => ({ ...prev, [name]: value }));
  };

  const handleCategoryChange = (value: string) => {
    setFormData((prev) => ({ ...prev, category: value }));
  };

  const handleStatusChange = (value: string) => {
    setFormData((prev) => ({ ...prev, status: value }));
  };

  const handleCreatePolicy = async (e: React.FormEvent) => {
    e.preventDefault();
    
    // Validate inputs
    if (!formData.name || !formData.category || !formData.status) {
      toast.error("All fields are required");
      return;
    }
    
    setSubmitting(true);
    
    try {
      const isSuperAdmin = session?.user?.role === 'super_admin';
      const headers = addHospitalIdHeader({ 
        'Content-Type': 'application/json',
      }, Boolean(isSuperAdmin));
      
      const response = await fetch("/api/compliance/policies", {
        method: "POST",
        headers,
        body: JSON.stringify({
          name: formData.name,
          category: formData.category,
          status: formData.status,
        }),
      });
      
      if (!response.ok) {
        const errorData = await response.json();
        console.error("API Error:", errorData);
        throw new Error(errorData.error || errorData.message || "Failed to create policy");
      }
      
      toast.success("Policy created successfully");
      setFormData({ name: "", category: "", status: "" });
      setShowForm(false);
      refetch();
    } catch (err) {
      console.error("Error creating policy:", err);
      toast.error((err as Error).message);
    } finally {
      setSubmitting(false);
    }
  };
  
  if (loading || statusesLoading || categoriesLoading) {
    return <PoliciesTabSkeleton />;
  }
  
  if (error) {
    return <div className="p-4 text-red-500">Error loading policies data: {error}</div>;
  }

  const getStatusClass = (status: string) => {
    switch (status.toLowerCase()) {
      case 'active':
        return 'text-green-600';
      case 'review due':
        return 'text-amber-600';
      case 'update required':
        return 'text-red-600';
      default:
        return 'text-gray-600';
    }
  };

  return (
    <Card>
      <CardHeader className="flex flex-row items-center justify-between pb-2">
        <div>
          <CardTitle>Compliance Policies</CardTitle>
          <CardDescription>Manage organizational compliance policies</CardDescription>
        </div>
        <Button 
          variant="outline" 
          size="sm" 
          onClick={() => setShowForm(!showForm)}
          className="flex items-center gap-1"
        >
          {showForm ? <X size={16} /> : <Plus size={16} />}
          {showForm ? "Cancel" : "Add Policy"}
        </Button>
      </CardHeader>
      <CardContent>
        <div className="space-y-4">
          {showForm && (
            <Card className="border-dashed border-2 border-blue-200 bg-blue-50 mb-4">
              <CardContent className="pt-4">
                <form onSubmit={handleCreatePolicy} className="space-y-4">
                  <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                    <div>
                      <label htmlFor="name" className="block text-sm font-medium mb-1">Policy Name</label>
                      <Input
                        id="name"
                        name="name"
                        placeholder="e.g. Patient Data Access"
                        value={formData.name}
                        onChange={handleInputChange}
                        className="w-full"
                      />
                    </div>
                    <div>
                      <label htmlFor="category" className="block text-sm font-medium mb-1">Category</label>
                      <Select value={formData.category} onValueChange={handleCategoryChange}>
                        <SelectTrigger className="w-full">
                          <SelectValue placeholder="Select Category" />
                        </SelectTrigger>
                        <SelectContent>
                          {policyCategories.map(category => (
                            <SelectItem key={category.value} value={category.value}>
                              {category.label}
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                    </div>
                    <div>
                      <label htmlFor="status" className="block text-sm font-medium mb-1">Status</label>
                      <Select value={formData.status} onValueChange={handleStatusChange}>
                        <SelectTrigger className="w-full">
                          <SelectValue placeholder="Select Status" />
                        </SelectTrigger>
                        <SelectContent>
                          {policyStatuses.map(status => (
                            <SelectItem key={status.value} value={status.value}>
                              {status.label}
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                    </div>
                  </div>
                  <div className="flex justify-end">
                    <Button 
                      type="submit" 
                      disabled={submitting}
                      className="bg-blue-600 hover:bg-blue-700"
                    >
                      {submitting ? "Creating..." : "Create Policy"}
                    </Button>
                  </div>
                </form>
              </CardContent>
            </Card>
          )}

          <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
            {Object.keys(policies).length > 0 ? (
              Object.entries(policies).map(([category, items]) => (
                <div key={category} className="border rounded-md p-4">
                  <h3 className="font-medium mb-2">{category}</h3>
                  <ul className="space-y-1 text-sm">
                    {items.map(policy => (
                      <li key={policy.id} className="flex justify-between">
                        <span>{policy.name}</span> 
                        <span className={getStatusClass(policy.status)}>{policy.status}</span>
                      </li>
                    ))}
                  </ul>
                </div>
              ))
            ) : (
              <div className="col-span-3 text-center py-4 text-gray-500">
                No policies available
              </div>
            )}
          </div>
        </div>
      </CardContent>
    </Card>
  );
};

const PoliciesTabSkeleton = () => {
  return (
    <Card>
      <CardHeader>
        <CardTitle>Compliance Policies</CardTitle>
        <CardDescription>Manage organizational compliance policies</CardDescription>
      </CardHeader>
      <CardContent>
        <div className="space-y-4">
          <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
            {[1, 2, 3].map(i => (
              <div key={i} className="border rounded-md p-4">
                <h3 className="font-medium mb-2"><Skeleton className="h-5 w-24" /></h3>
                <ul className="space-y-1 text-sm">
                  {[1, 2, 3].map(j => (
                    <li key={j} className="flex justify-between">
                      <Skeleton className="h-4 w-20" />
                      <Skeleton className="h-4 w-10" />
                    </li>
                  ))}
                </ul>
              </div>
            ))}
          </div>
          <Skeleton className="h-10 w-32" />
        </div>
      </CardContent>
    </Card>
  );
};

export default PoliciesTab; 