import { NextResponse } from "next/server";
import { auth } from '@/lib/auth';
import prisma from "@/lib/prisma";

// GET: Fetch all training events for the user's hospital
export async function GET(request: Request) {
  try {
    const session = await auth();
    
    if (!session?.user?.email) {
      return NextResponse.json({ error: "Unauthorized" }, { status: 401 });
    }
    
    // Get the hospital ID from the user session
    const user = await prisma.user.findUnique({
      where: {
        email: session.user.email,
      },
      select: {
        hospital_id: true,
        role: true,
      },
    });
    
    if (!user) {
      return NextResponse.json({ error: "User not found" }, { status: 404 });
    }

    // Handle hospital ID - for superadmins, get from cookie/header
    let hospitalId = user.hospital_id;
    
    if (!hospitalId && user.role === 'super_admin') {
      // Get hospital ID from request header or cookie
      const headerHospitalId = request.headers.get('x-hospital-id');
      const cookieHeader = request.headers.get('cookie');
      let cookieHospitalId = null;
      
      if (cookieHeader) {
        const cookies = cookieHeader.split(';').map(c => c.trim());
        const hospitalCookie = cookies.find(c => c.startsWith('selectedHospitalId='));
        if (hospitalCookie) {
          cookieHospitalId = hospitalCookie.split('=')[1];
        }
      }
      
      const hospitalIdStr = headerHospitalId || cookieHospitalId;
      
      if (hospitalIdStr) {
        hospitalId = parseInt(hospitalIdStr);
      }
    }

    // If no hospital ID available, return empty array
    if (!hospitalId) {
      return NextResponse.json([]);
    }
    
    // Fetch training events for the hospital
    const trainingEvents = await prisma.trainingEvent.findMany({
      where: {
        hospital_id: hospitalId,
      },
      orderBy: {
        date: "asc",
      },
    });
    
    return NextResponse.json(trainingEvents);
  } catch (error) {
    console.error("Error fetching training events:", error);
    return NextResponse.json({ error: "Failed to fetch training events" }, { status: 500 });
  }
}

// POST: Create a new training event
export async function POST(request: Request) {
  try {
    console.log('=== Training API POST - Start ===');
    const session = await auth();
    
    console.log('Training API - Session:', session ? 'exists' : 'null');
    console.log('Training API - User email:', session?.user?.email);
    
    if (!session?.user?.email) {
      console.log('Training API - Error: No session or email');
      return NextResponse.json({ error: "Unauthorized" }, { status: 401 });
    }
    
    console.log('Training API - Fetching user from database...');
    // Get the hospital ID from the user session
    const user = await prisma.user.findUnique({
      where: {
        email: session.user.email,
      },
      select: {
        hospital_id: true,
        role: true,
      },
    });
    
    console.log('Training API - User from DB:', user);
    
    if (!user) {
      console.log('Training API - Error: User not found in database');
      return NextResponse.json({ error: "User not found" }, { status: 404 });
    }

    // Handle hospital ID - for superadmins, get from cookie/header
    let hospitalId = user.hospital_id;
    
    if (!hospitalId && user.role === 'super_admin') {
      console.log('Training API - User is superadmin, checking for selected hospital...');
      
      // Get hospital ID from request header or cookie
      const headerHospitalId = request.headers.get('x-hospital-id');
      const cookieHeader = request.headers.get('cookie');
      let cookieHospitalId = null;
      
      if (cookieHeader) {
        const cookies = cookieHeader.split(';').map(c => c.trim());
        const hospitalCookie = cookies.find(c => c.startsWith('selectedHospitalId='));
        if (hospitalCookie) {
          cookieHospitalId = hospitalCookie.split('=')[1];
        }
      }
      
      const hospitalIdStr = headerHospitalId || cookieHospitalId;
      
      if (hospitalIdStr) {
        hospitalId = parseInt(hospitalIdStr);
        console.log('Training API - Using selected hospital ID:', hospitalId);
      }
    }
    
    if (!hospitalId) {
      console.log('Training API - Error: No hospital_id available');
      return NextResponse.json({ 
        error: "No hospital selected",
        details: "Please select a hospital before creating training events"
      }, { status: 400 });
    }
    
    console.log('Training API - Parsing request data...');
    const data = await request.json();
    console.log('Training API - Request data:', JSON.stringify(data, null, 2));
    
    // Validate required fields
    console.log('Training API - Starting field validation...');
    const requiredFields = ["title", "date", "instructor", "location", "department", "duration", "capacity", "enrolled"];
    const missingFields = [];
    
    for (const field of requiredFields) {
      console.log(`Training API - Checking field ${field}:`, data[field], typeof data[field]);
      if (data[field] === undefined || data[field] === null || data[field] === '') {
        missingFields.push(field);
        console.log(`Training API - Missing/empty field: ${field}, value:`, data[field]);
      }
    }
    
    if (missingFields.length > 0) {
      const errorMessage = `Missing or empty required fields: ${missingFields.join(', ')}`;
      console.log('Training API - Validation error:', errorMessage);
      return NextResponse.json({ 
        error: errorMessage,
        details: `Please ensure all required fields are filled: ${missingFields.join(', ')}`
      }, { status: 400 });
    }
    
    console.log('Training API - Field validation passed, checking numeric values...');
    
    // Additional validation
    if (isNaN(parseInt(data.capacity)) || parseInt(data.capacity) <= 0) {
      console.log('Training API - Error: Invalid capacity:', data.capacity);
      return NextResponse.json({ 
        error: "Invalid capacity value",
        details: "Capacity must be a positive number"
      }, { status: 400 });
    }
    
    if (isNaN(parseInt(data.enrolled)) || parseInt(data.enrolled) < 0) {
      console.log('Training API - Error: Invalid enrolled:', data.enrolled);
      return NextResponse.json({ 
        error: "Invalid enrolled value",
        details: "Enrolled must be a non-negative number"
      }, { status: 400 });
    }
    
    if (parseInt(data.enrolled) > parseInt(data.capacity)) {
      console.log('Training API - Error: Enrolled > capacity:', data.enrolled, '>', data.capacity);
      return NextResponse.json({ 
        error: "Invalid enrollment data",
        details: "Enrolled count cannot exceed capacity"
      }, { status: 400 });
    }
    
    console.log('Training API - All validation passed, creating training event...');
    
    // Create the training event with the hospital ID
    const trainingEvent = await prisma.trainingEvent.create({
      data: {
        title: data.title,
        date: new Date(data.date),
        instructor: data.instructor,
        location: data.location,
        department: data.department,
        duration: data.duration,
        capacity: parseInt(data.capacity),
        enrolled: parseInt(data.enrolled),
        required: Boolean(data.required),
        hospital_id: hospitalId,
      },
    });
    
    console.log('Training API - Created training event successfully:', trainingEvent);
    console.log('=== Training API POST - Success ===');
    return NextResponse.json(trainingEvent, { status: 201 });
  } catch (error) {
    console.error("=== Training API POST - Error ===");
    console.error("Error creating training event:", error);
    console.error("Error stack:", error instanceof Error ? error.stack : 'No stack trace');
    console.error("=== End Error Details ===");
    return NextResponse.json({ error: "Failed to create training event" }, { status: 500 });
  }
}

// PATCH: Update an existing training event
export async function PATCH(request: Request) {
  try {
    const session = await auth();
    
    if (!session?.user?.email) {
      return NextResponse.json({ error: "Unauthorized" }, { status: 401 });
    }
    
    const data = await request.json();
    
    if (!data.id) {
      return NextResponse.json({ error: "Missing training event ID" }, { status: 400 });
    }
    
    // Get the hospital ID from the user session
    const user = await prisma.user.findUnique({
      where: {
        email: session.user.email,
      },
      select: {
        hospital_id: true,
        role: true,
      },
    });
    
    if (!user) {
      return NextResponse.json({ error: "User not found" }, { status: 404 });
    }

    // Handle hospital ID - for superadmins, get from cookie/header
    let hospitalId = user.hospital_id;
    
    if (!hospitalId && user.role === 'super_admin') {
      // Get hospital ID from request header or cookie
      const headerHospitalId = request.headers.get('x-hospital-id');
      const cookieHeader = request.headers.get('cookie');
      let cookieHospitalId = null;
      
      if (cookieHeader) {
        const cookies = cookieHeader.split(';').map(c => c.trim());
        const hospitalCookie = cookies.find(c => c.startsWith('selectedHospitalId='));
        if (hospitalCookie) {
          cookieHospitalId = hospitalCookie.split('=')[1];
        }
      }
      
      const hospitalIdStr = headerHospitalId || cookieHospitalId;
      
      if (hospitalIdStr) {
        hospitalId = parseInt(hospitalIdStr);
      }
    }
    
    if (!hospitalId) {
      return NextResponse.json({ 
        error: "No hospital selected",
        details: "Please select a hospital before updating training events"
      }, { status: 400 });
    }
    
    // Verify the training event belongs to the user's hospital
    const existingEvent = await prisma.trainingEvent.findUnique({
      where: {
        id: data.id,
      },
    });
    
    if (!existingEvent) {
      return NextResponse.json({ error: "Training event not found" }, { status: 404 });
    }
    
    // Check hospital ownership
    if (existingEvent.hospital_id !== hospitalId) {
      return NextResponse.json({ error: "Unauthorized to modify this training event" }, { status: 403 });
    }
    
    // Prepare update data
    const updateData: any = { 
      title: data.title,
      instructor: data.instructor,
      location: data.location,
      department: data.department,
      duration: data.duration,
    };
    
    // Format date if present
    if (data.date) {
      updateData.date = new Date(data.date);
    }
    
    // Update numeric fields if present
    if (data.capacity !== undefined) {
      updateData.capacity = parseInt(data.capacity);
    }
    
    if (data.enrolled !== undefined) {
      updateData.enrolled = parseInt(data.enrolled);
    }
    
    // Update boolean field if present
    if (data.required !== undefined) {
      updateData.required = Boolean(data.required);
    }
    
    // Update the training event
    const updatedEvent = await prisma.trainingEvent.update({
      where: {
        id: data.id,
      },
      data: updateData,
    });
    
    return NextResponse.json(updatedEvent);
  } catch (error) {
    console.error("Error updating training event:", error);
    return NextResponse.json({ error: "Failed to update training event" }, { status: 500 });
  }
}

// DELETE: Delete a training event
export async function DELETE(request: Request) {
  try {
    const session = await auth();
    
    if (!session?.user?.email) {
      return NextResponse.json({ error: "Unauthorized" }, { status: 401 });
    }
    
    const { searchParams } = new URL(request.url);
    const id = searchParams.get("id");
    
    if (!id) {
      return NextResponse.json({ error: "Missing training event ID" }, { status: 400 });
    }
    
    // Get the hospital ID from the user session
    const user = await prisma.user.findUnique({
      where: {
        email: session.user.email,
      },
      select: {
        hospital_id: true,
        role: true,
      },
    });
    
    if (!user) {
      return NextResponse.json({ error: "User not found" }, { status: 404 });
    }

    // Handle hospital ID - for superadmins, get from cookie/header
    let hospitalId = user.hospital_id;
    
    if (!hospitalId && user.role === 'super_admin') {
      // Get hospital ID from request header or cookie
      const headerHospitalId = request.headers.get('x-hospital-id');
      const cookieHeader = request.headers.get('cookie');
      let cookieHospitalId = null;
      
      if (cookieHeader) {
        const cookies = cookieHeader.split(';').map(c => c.trim());
        const hospitalCookie = cookies.find(c => c.startsWith('selectedHospitalId='));
        if (hospitalCookie) {
          cookieHospitalId = hospitalCookie.split('=')[1];
        }
      }
      
      const hospitalIdStr = headerHospitalId || cookieHospitalId;
      
      if (hospitalIdStr) {
        hospitalId = parseInt(hospitalIdStr);
      }
    }
    
    if (!hospitalId) {
      return NextResponse.json({ 
        error: "No hospital selected",
        details: "Please select a hospital before deleting training events"
      }, { status: 400 });
    }
    
    // Verify the training event belongs to the user's hospital
    const existingEvent = await prisma.trainingEvent.findUnique({
      where: {
        id,
      },
    });
    
    if (!existingEvent) {
      return NextResponse.json({ error: "Training event not found" }, { status: 404 });
    }
    
    // Check hospital ownership
    if (existingEvent.hospital_id !== hospitalId) {
      return NextResponse.json({ error: "Unauthorized to delete this training event" }, { status: 403 });
    }
    
    // Delete the training event
    await prisma.trainingEvent.delete({
      where: {
        id,
      },
    });
    
    return NextResponse.json({ success: true });
  } catch (error) {
    console.error("Error deleting training event:", error);
    return NextResponse.json({ error: "Failed to delete training event" }, { status: 500 });
  }
} 