import { NextRequest, NextResponse } from 'next/server';
import { getApiContext, handleApiError } from '@/lib/api-helpers';
import prisma from '@/lib/prisma';

// GET - fetch all staff compliance data for a hospital
export async function GET(req: NextRequest) {
  try {
    // Get user context (hospital ID)
    const { hospitalId, isSuperAdmin, user } = await getApiContext(req);
    
    // Ensure we have a hospital ID
    if (!hospitalId) {
      return NextResponse.json({ 
        error: 'Missing hospital ID',
        message: 'A hospital ID is required to fetch staff compliance data' 
      }, { status: 400 });
    }
    
    console.log(`Fetching staff compliance data for hospital: ${hospitalId}`);
    
    // Convert to number for database query
    const hospitalIdNum = Number(hospitalId);
    
    // Fetch staff compliance records with simple query
    const staffCompliance = await prisma.staffCompliance.findMany({
      where: {
        hospital_id: hospitalIdNum
      },
      orderBy: {
        department: 'asc'
      },
      select: {
        id: true,
        department: true,
        staff_count: true,
        compliance_rate: true,
        notes: true,
        hospital_id: true,
        created_at: true,
        updated_at: true
      }
    });
    
    // Calculate overall compliance rate if we have records
    let overallRate = 0;
    if (staffCompliance.length > 0) {
      const totalStaff = staffCompliance.reduce((sum: number, dept: any) => sum + dept.staff_count, 0);
      const weightedCompliance = staffCompliance.reduce((sum: number, dept: any) => 
        sum + (dept.compliance_rate * dept.staff_count), 0);
        
      overallRate = totalStaff > 0 ? Math.round(weightedCompliance / totalStaff) : 0;
    }
    
    // Format the response to match our frontend structure
    const formattedData = {
      overallRate,
      departments: staffCompliance.map((item: any) => ({
        id: item.id,
        name: item.department,
        complianceRate: item.compliance_rate,
        staffCount: item.staff_count,
        lastUpdated: item.updated_at
      }))
    };
    
    return NextResponse.json(formattedData);
    
  } catch (error) {
    console.error("Error fetching staff compliance data:", error instanceof Error ? error.message : String(error));
    return handleApiError(error);
  }
}

// POST - create a new staff compliance record
export async function POST(req: Request) {
  try {
    console.log('Staff Compliance API - Starting POST request');
    
    // Get user context
    const { hospitalId, isSuperAdmin, user } = await getApiContext(req);
    
    console.log('Staff Compliance API - User context:', { hospitalId, isSuperAdmin, userEmail: user?.email });
    
    // Ensure we have a hospital ID
    if (!hospitalId) {
      console.log('Staff Compliance API - Missing hospital ID');
      return NextResponse.json({ 
        error: 'Missing hospital ID',
        message: 'A hospital ID is required to create staff compliance data' 
      }, { status: 400 });
    }
    
    // Parse request body
    const data = await req.json();
    console.log('Staff Compliance API - Request data:', data);
    
    // Validate required fields
    if (!data.department || !data.staffCount || data.complianceRate === undefined) {
      console.log('Staff Compliance API - Missing required fields', { 
        hasDepartment: !!data.department, 
        hasStaffCount: !!data.staffCount, 
        hasComplianceRate: data.complianceRate !== undefined 
      });
      return NextResponse.json({ 
        error: 'Missing required fields',
        message: 'Department, staff count, and compliance rate are required' 
      }, { status: 400 });
    }
    
    // Convert to number for database
    const hospitalIdNum = Number(hospitalId);
    
    // Check if record already exists for this department
    const existingRecord = await prisma.staffCompliance.findFirst({
      where: {
        hospital_id: hospitalIdNum,
        department: data.department
      }
    });
    
    if (existingRecord) {
      console.log('Staff Compliance API - Department already exists:', data.department);
      return NextResponse.json({ 
        error: 'Record exists',
        message: 'A compliance record for this department already exists' 
      }, { status: 409 });
    }
    
    // Create new record
    try {
      const newRecord = await prisma.staffCompliance.create({
        data: {
          hospital_id: hospitalIdNum,
          department: data.department,
          staff_count: data.staffCount,
          compliance_rate: data.complianceRate,
          notes: data.notes || null
        }
      });
      
      console.log('Staff Compliance API - Record created:', newRecord);
      
      return NextResponse.json({
        success: true,
        id: newRecord.id,
        message: 'Staff compliance record created successfully'
      }, { status: 201 });
    } catch (dbError) {
      console.error('Staff Compliance API - Database error:', dbError);
      // Return a proper JSON error response instead of re-throwing
      return NextResponse.json({
        error: 'Database error',
        message: dbError instanceof Error ? dbError.message : 'An error occurred while creating the record',
        details: dbError
      }, { status: 500 });
    }
  } catch (error) {
    console.error("Error creating staff compliance record:", error instanceof Error ? error.message : String(error));
    return handleApiError(error);
  }
}

// PATCH - update an existing staff compliance record
export async function PATCH(req: Request) {
  try {
    // Get user context
    const { hospitalId, isSuperAdmin, user } = await getApiContext(req);
    
    // Ensure we have a hospital ID
    if (!hospitalId) {
      return NextResponse.json({ 
        error: 'Missing hospital ID',
        message: 'A hospital ID is required to update staff compliance data' 
      }, { status: 400 });
    }
    
    // Parse request body
    const data = await req.json();
    
    // Validate record ID
    if (!data.id) {
      return NextResponse.json({ 
        error: 'Missing record ID',
        message: 'Record ID is required for updates' 
      }, { status: 400 });
    }
    
    // Convert to number for database
    const hospitalIdNum = Number(hospitalId);
    
    // Check if record exists and belongs to this hospital
    const existingRecord = await prisma.staffCompliance.findUnique({
      where: {
        id: data.id
      }
    });
    
    if (!existingRecord) {
      return NextResponse.json({ 
        error: 'Record not found',
        message: 'The specified staff compliance record does not exist' 
      }, { status: 404 });
    }
    
    if (existingRecord.hospital_id !== hospitalIdNum) {
      return NextResponse.json({ 
        error: 'Unauthorized',
        message: 'You do not have permission to update this record' 
      }, { status: 403 });
    }
    
    // Update the record
    try {
      const updatedRecord = await prisma.staffCompliance.update({
        where: {
          id: data.id
        },
        data: {
          department: data.department || existingRecord.department,
          staff_count: data.staffCount ?? existingRecord.staff_count,
          compliance_rate: data.complianceRate ?? existingRecord.compliance_rate,
          notes: data.notes ?? existingRecord.notes
        }
      });
      
      console.log('Staff Compliance API - Record updated successfully:', updatedRecord);
      
      return NextResponse.json({
        success: true,
        id: updatedRecord.id,
        message: 'Staff compliance record updated successfully'
      });
    } catch (dbError) {
      console.error('Staff Compliance API - Database error during update:', dbError);
      return NextResponse.json({
        error: 'Database error',
        message: dbError instanceof Error ? dbError.message : 'An error occurred while updating the record',
        details: dbError
      }, { status: 500 });
    }
  } catch (error) {
    console.error("Error updating staff compliance record:", error instanceof Error ? error.message : String(error));
    return handleApiError(error);
  }
}

// DELETE - delete a staff compliance record
export async function DELETE(req: Request) {
  try {
    console.log('Staff Compliance API - Starting DELETE request');
    
    // Get user context
    const { hospitalId, isSuperAdmin, user } = await getApiContext(req);
    
    console.log('Staff Compliance API - User context:', { hospitalId, isSuperAdmin, userEmail: user?.email });
    
    // Ensure we have a hospital ID
    if (!hospitalId) {
      console.log('Staff Compliance API - Missing hospital ID');
      return NextResponse.json({ 
        error: 'Missing hospital ID',
        message: 'A hospital ID is required to delete staff compliance data' 
      }, { status: 400 });
    }
    
    // Get record ID from URL
    const url = new URL(req.url);
    const id = url.searchParams.get('id');
    
    console.log('Staff Compliance API - Delete request for ID:', id);
    
    if (!id) {
      console.log('Staff Compliance API - Missing record ID');
      return NextResponse.json({ 
        error: 'Missing record ID',
        message: 'Record ID is required for deletion' 
      }, { status: 400 });
    }
    
    // Convert to number for database
    const hospitalIdNum = Number(hospitalId);
    
    // Check if record exists and belongs to this hospital
    const existingRecord = await prisma.staffCompliance.findUnique({
      where: {
        id: Number(id)
      }
    });
    
    console.log('Staff Compliance API - Found record:', existingRecord);
    
    if (!existingRecord) {
      console.log('Staff Compliance API - Record not found');
      return NextResponse.json({ 
        error: 'Record not found',
        message: 'The specified staff compliance record does not exist' 
      }, { status: 404 });
    }
    
    if (existingRecord.hospital_id !== hospitalIdNum) {
      console.log('Staff Compliance API - Unauthorized deletion attempt, record belongs to hospital', existingRecord.hospital_id);
      return NextResponse.json({ 
        error: 'Unauthorized',
        message: 'You do not have permission to delete this record' 
      }, { status: 403 });
    }
    
    // Delete the record
    try {
      await prisma.staffCompliance.delete({
        where: {
          id: Number(id)
        }
      });
      
      console.log('Staff Compliance API - Record deleted successfully');
      
      return NextResponse.json({
        success: true,
        message: 'Staff compliance record deleted successfully'
      });
    } catch (dbError) {
      console.error('Staff Compliance API - Database error during deletion:', dbError);
      // Return a proper JSON error response instead of re-throwing
      return NextResponse.json({
        error: 'Database error',
        message: dbError instanceof Error ? dbError.message : 'An error occurred while deleting the record',
        details: dbError
      }, { status: 500 });
    }
  } catch (error) {
    console.error("Error deleting staff compliance record:", error instanceof Error ? error.message : String(error));
    return handleApiError(error);
  }
} 