import { NextResponse } from "next/server";
import { auth } from "@/lib/auth";
import prisma from "@/lib/prisma";

// GET: Fetch all audit schedules for the user's hospital
export async function GET() {
  console.log("GET /api/schedule/audits.ts - Start");
  try {
    const session = await auth();
    console.log("Session:", session ? "exists" : "null");
    
    if (!session || !session.user) {
      console.log("Unauthorized - No session");
      return NextResponse.json({ error: "Unauthorized" }, { status: 401 });
    }
    
    // Get the hospital ID from the user session
    console.log("Getting user hospital ID for:", session.user.email);
    const user = await prisma.user.findUnique({
      where: {
        email: session.user.email as string,
      },
      select: {
        hospital_id: true,
      },
    });
    
    console.log("User data:", user);
    
    if (!user || !user.hospital_id) {
      console.log("No hospital associated with user");
      return NextResponse.json({ error: "No hospital associated with user" }, { status: 404 });
    }
    
    // Fetch audit schedules for the hospital
    console.log("Fetching audit schedules for hospital:", user.hospital_id);
    const auditSchedules = await prisma.auditSchedule.findMany({
      where: {
        hospital_id: user.hospital_id,
      },
      orderBy: {
        date: "asc",
      },
    });
    
    console.log("Found audit schedules:", auditSchedules.length);
    return NextResponse.json(auditSchedules);
  } catch (error) {
    console.error("Error fetching audit schedules:", error);
    return NextResponse.json({ error: "Failed to fetch audit schedules" }, { status: 500 });
  }
}

// POST: Create a new audit schedule
export async function POST(request: Request) {
  try {
    const session = await auth();
    
    if (!session || !session.user) {
      return NextResponse.json({ error: "Unauthorized" }, { status: 401 });
    }
    
    // Get the hospital ID from the user session
    const user = await prisma.user.findUnique({
      where: {
        email: session.user.email as string,
      },
      select: {
        hospital_id: true,
      },
    });
    
    if (!user || !user.hospital_id) {
      return NextResponse.json({ error: "No hospital associated with user" }, { status: 404 });
    }
    
    const data = await request.json();
    
    // Validate required fields
    const requiredFields = ["title", "date", "department", "auditor", "status"];
    for (const field of requiredFields) {
      if (!data[field]) {
        return NextResponse.json({ error: `Missing required field: ${field}` }, { status: 400 });
      }
    }
    
    // Create the audit schedule with the hospital ID
    const auditSchedule = await prisma.auditSchedule.create({
      data: {
        title: data.title,
        date: new Date(data.date),
        department: data.department,
        auditor: data.auditor,
        status: data.status,
        hospital_id: user.hospital_id,
      },
    });
    
    return NextResponse.json(auditSchedule, { status: 201 });
  } catch (error) {
    console.error("Error creating audit schedule:", error);
    return NextResponse.json({ error: "Failed to create audit schedule" }, { status: 500 });
  }
} 