import { NextResponse } from 'next/server';
import fs from 'fs';
import path from 'path';

export async function GET(req: Request) {
  try {
    // Parse URL to get file path
    const url = new URL(req.url);
    const filePath = url.searchParams.get('file');
    
    console.log('Download request for file:', filePath);
    
    if (!filePath) {
      return NextResponse.json({ 
        error: 'Missing file parameter',
      }, { status: 400 });
    }
    
    // Basic sanity check - make sure the path isn't malicious
    if (filePath.includes('..') || filePath.includes('\\')) {
      return NextResponse.json({ 
        error: 'Invalid file path',
      }, { status: 400 });
    }
    
    // Simple approach - allow any file in the public directory
    // Strip any leading slashes for consistency
    const normalizedPath = filePath.replace(/^\/+/, '');
    
    console.log('Normalized path:', normalizedPath);
    
    // Get the full path to the file in the public directory
    const fullPath = path.join(process.cwd(), 'public', normalizedPath);
    
    console.log('Full path:', fullPath);
    
    // Check if the file exists
    if (!fs.existsSync(fullPath)) {
      console.error(`File not found: ${fullPath}`);
      
      // Try to find similar files to help with debugging
      const dir = path.dirname(fullPath);
      if (fs.existsSync(dir)) {
        const files = fs.readdirSync(dir);
        console.log(`Files in ${dir}:`, files);
        
        // Check if there's a .txt version when a .pdf file was requested
        if (fullPath.endsWith('.pdf')) {
          const txtPath = fullPath.replace(/\.pdf$/i, '.txt');
          if (fs.existsSync(txtPath)) {
            console.log(`Found alternative txt file: ${txtPath}`);
            return new NextResponse(fs.readFileSync(txtPath), {
              status: 200,
              headers: {
                'Content-Type': 'text/plain',
                'Content-Disposition': `attachment; filename="${path.basename(txtPath)}"`
              },
            });
          }
        }
        
        // Try to find a similar file if we're looking for something with a timestamp
        const baseName = path.basename(normalizedPath);
        const filePattern = baseName.replace(/\d{4}-\d{2}-\d{2}T\d{2}-\d{2}-\d{2}-\d{3}Z/i, '');
        
        if (filePattern !== baseName) {
          // We have a timestamp pattern, try to find similar files
          console.log(`Looking for files similar to pattern: ${filePattern}`);
          
          // Find any files that start with the same prefix (before the timestamp)
          const similarFiles = files.filter(file => 
            file.startsWith(filePattern.split('T')[0]) && 
            (file.endsWith('.txt') || file.endsWith('.pdf'))
          );
          
          if (similarFiles.length > 0) {
            // Sort by name descending to get the most recent one
            similarFiles.sort().reverse();
            const mostRecentFile = similarFiles[0];
            const mostRecentPath = path.join(dir, mostRecentFile);
            
            console.log(`Found similar file: ${mostRecentPath}`);
            
            // Determine content type
            const contentType = mostRecentFile.endsWith('.pdf') ? 
              'application/pdf' : 'text/plain';
            
            return new NextResponse(fs.readFileSync(mostRecentPath), {
              status: 200,
              headers: {
                'Content-Type': contentType,
                'Content-Disposition': `attachment; filename="${mostRecentFile}"`
              },
            });
          }
        }
      } else {
        console.log(`Directory doesn't exist: ${dir}`);
      }
      
      return NextResponse.json({ 
        error: 'File not found',
        path: normalizedPath
      }, { status: 404 });
    }
    
    // Read the file
    const fileContent = fs.readFileSync(fullPath);
    
    // Determine content type based on file extension
    const ext = path.extname(fullPath).toLowerCase();
    let contentType = 'application/octet-stream'; // Default content type
    
    switch (ext) {
      case '.pdf':
        contentType = 'application/pdf';
        break;
      case '.txt':
        contentType = 'text/plain';
        break;
      case '.html':
        contentType = 'text/html';
        break;
      case '.json':
        contentType = 'application/json';
        break;
      // Add more types as needed
    }
    
    // Create headers for the response
    const headers = new Headers();
    headers.set('Content-Type', contentType);
    headers.set('Content-Disposition', `attachment; filename="${path.basename(fullPath)}"`);
    
    // Return the file with appropriate headers
    return new NextResponse(fileContent, {
      status: 200,
      headers,
    });
  } catch (error) {
    console.error("Error serving file:", error);
    return NextResponse.json({ 
      error: 'Failed to serve file',
      message: error instanceof Error ? error.message : 'An unexpected error occurred',
    }, { status: 500 });
  }
} 