import { NextResponse } from "next/server"
import { auth } from "@/lib/auth"
import prisma from "@/lib/prisma"

// GET: Get the hospital profile
export async function GET() {
  try {
    const session = await auth();
    
    if (!session || !session.user) {
      return NextResponse.json({ error: "Unauthorized" }, { status: 401 })
    }
    
    // Get the hospital ID from the user session
    const user = await prisma.user.findUnique({
      where: {
        email: session.user.email,
      },
      select: {
        hospital_id: true,
      },
    })
    
    if (!user || !user.hospital_id) {
      return NextResponse.json({ error: "No hospital associated with user" }, { status: 404 })
    }
    
    // Fetch hospital profile
    const hospital = await prisma.hospital.findUnique({
      where: {
        id: user.hospital_id,
      },
      select: {
        id: true,
        name: true,
        location: true,
        logo_url: true,
      },
    })
    
    if (!hospital) {
      return NextResponse.json({ error: "Hospital not found" }, { status: 404 })
    }
    
    return NextResponse.json(hospital)
  } catch (error) {
    console.error("Error fetching hospital profile:", error)
    return NextResponse.json({ error: "Failed to fetch hospital profile" }, { status: 500 })
  }
}

// PUT: Update the hospital profile
export async function PUT(request: Request) {
  try {
    const session = await auth();
    
    if (!session || !session.user) {
      return NextResponse.json({ error: "Unauthorized" }, { status: 401 })
    }
    
    // Get the hospital ID from the user session
    const user = await prisma.user.findUnique({
      where: {
        email: session.user.email,
      },
      select: {
        hospital_id: true,
      },
    })
    
    if (!user || !user.hospital_id) {
      return NextResponse.json({ error: "No hospital associated with user" }, { status: 404 })
    }
    
    const data = await request.json()
    
    // Validate required fields
    if (!data.name || !data.location) {
      return NextResponse.json({ error: "Name and location are required" }, { status: 400 })
    }
    
    // Update the hospital profile
    const updatedHospital = await prisma.hospital.update({
      where: {
        id: user.hospital_id,
      },
      data: {
        name: data.name,
        location: data.location,
        logo_url: data.logo_url,
      },
    })
    
    return NextResponse.json(updatedHospital)
  } catch (error) {
    console.error("Error updating hospital profile:", error)
    return NextResponse.json({ error: "Failed to update hospital profile" }, { status: 500 })
  }
} 