import { NextRequest, NextResponse } from 'next/server';
import { auth } from '@/lib/auth';
import prisma from '@/lib/prisma';
import { getApiContext, handleApiError } from '@/lib/api-helpers';

export async function GET(request: NextRequest) {
  try {
    // Use our new helper to get context including hospital ID
    const { user, isSuperAdmin, hospitalId } = await getApiContext(request);
    
    // For debugging
    console.log(`Findings API - Hospital ID: ${hospitalId}, Super Admin: ${isSuperAdmin}`);
    
    // If super admin with no hospital selected, return an error
    if (isSuperAdmin && !hospitalId) {
      return NextResponse.json({ 
        error: 'Hospital selection required',
        message: 'Super admin must select a hospital to view findings data' 
      }, { status: 400 });
    }
    
    // Get audit findings for the selected/assigned hospital
    const findings = await prisma.auditFinding.findMany({
      where: hospitalId ? { hospital_id: hospitalId } : undefined,
      orderBy: { date_reported: 'desc' }
    });
    
    return NextResponse.json(findings);
  } catch (error) {
    return handleApiError(error);
  }
}

export async function POST(req: Request) {
  try {
    console.log("POST /api/findings - Starting request processing");
    
    // Use our new helper to get context including hospital ID
    try {
      var { user, isSuperAdmin, hospitalId } = await getApiContext(req);
      console.log(`Creating finding - User: ${user?.email}, Hospital ID: ${hospitalId}, Super Admin: ${isSuperAdmin}`);
    } catch (contextError) {
      console.error("Error getting API context:", contextError);
      
      // Special handling for the common case where regular users don't have a hospital
      const errorMessage = contextError instanceof Error ? contextError.message : 'Failed to authenticate user';
      
      if (errorMessage.includes('no assigned hospital')) {
        return NextResponse.json({ 
          error: 'Missing hospital assignment',
          message: 'Your user account is not assigned to any hospital. Please contact an administrator.'
        }, { status: 403 });
      }
      
      return NextResponse.json({ 
        error: 'Authentication error',
        message: errorMessage
      }, { status: 401 });
    }
    
    // If super admin with no hospital selected, return an error
    if (isSuperAdmin && !hospitalId) {
      console.log("Error: Super admin with no hospital selected");
      return NextResponse.json({ 
        error: 'Hospital selection required',
        message: 'Super admin must select a hospital to add findings data' 
      }, { status: 400 });
    }
    
    // Ensure we have a valid hospital ID
    if (!hospitalId) {
      console.log("Error: Missing hospital ID");
      return NextResponse.json({ 
        error: 'Missing hospital ID',
        message: 'A hospital ID is required to create a finding. Please select a hospital or ensure your account is properly configured.'
      }, { status: 400 });
    }
    
    // Parse request body
    let data;
    try {
      data = await req.json();
      console.log("Request data:", data);
    } catch (parseError) {
      console.error("Error parsing request JSON:", parseError);
      return NextResponse.json({ 
        error: 'Invalid JSON',
        message: 'The request body could not be parsed as JSON'
      }, { status: 400 });
    }
    
    // Validate the data
    if (!data.title || !data.department || !data.severity) {
      return NextResponse.json({ 
        error: 'Invalid data. Must include title, department, and severity fields.' 
      }, { status: 400 });
    }
    
    // Safely parse timeline date if present
    let timelineDate = null;
    if (data.timeline) {
      try {
        timelineDate = new Date(data.timeline);
        // Check if date is valid
        if (isNaN(timelineDate.getTime())) {
          timelineDate = null;
        }
      } catch (err) {
        console.warn("Invalid timeline date:", data.timeline);
      }
    }
    
    // Create the finding for the selected/assigned hospital
    try {
      console.log("Creating finding in database");
      const newFinding = await prisma.auditFinding.create({
        data: {
          title: data.title,
          department: data.department,
          severity: data.severity,
          status: data.status || 'open',
          description: data.description || null,
          date_reported: new Date(),
          resolution_date: null,
          timeline: timelineDate,
          hospital_id: hospitalId
        }
      });
      
      console.log("Finding created successfully:", newFinding);
      return NextResponse.json({ 
        message: 'Finding created successfully',
        finding: newFinding
      }, { status: 201 });
    } catch (dbError) {
      console.error('Database error creating finding:', dbError);
      return NextResponse.json({ 
        error: 'Database error',
        message: dbError instanceof Error ? dbError.message : 'Failed to create finding in database'
      }, { status: 500 });
    }
  } catch (error) {
    console.error('Unexpected error creating finding:', error);
    
    // Create a safe error object that can be JSON-serialized
    const safeError = error instanceof Error 
      ? { message: error.message, stack: error.stack, name: error.name } 
      : { message: String(error) };
      
    return NextResponse.json({ 
      error: 'Server error',
      message: error instanceof Error ? error.message : 'Failed to process request',
      details: safeError
    }, { status: 500 });
  }
} 