import { NextRequest, NextResponse } from 'next/server';
import prisma from '@/lib/prisma';
import { getApiContext, handleApiError } from '@/lib/api-helpers';

// GET /api/documentation/documents - Get all documents
export async function GET(request: NextRequest) {
  try {
    // Get context including hospital ID
    const { hospitalId } = await getApiContext(request);
    
    if (!hospitalId) {
      return NextResponse.json({ 
        error: 'Hospital selection required',
        message: 'A hospital must be selected to view documents data' 
      }, { status: 400 });
    }
    
    // Fetch documents for the hospital
    const documents = await prisma.document.findMany({
      where: { hospital_id: hospitalId },
      select: {
        id: true,
        title: true,
        category_id: true,
        category: true,
        status: true,
        department: true,
        fileName: true,
        fileSize: true,
        fileType: true,
        updatedBy: true,
        updatedAt: true,
        createdAt: true
      },
      orderBy: { updatedAt: 'desc' }
    });
    
    return NextResponse.json(documents);
  } catch (error) {
    return handleApiError(error);
  }
}

// POST /api/documentation/documents - Create a new document
export async function POST(request: Request) {
  try {
    const { user, hospitalId } = await getApiContext(request);
    
    if (!hospitalId) {
      return NextResponse.json({ 
        error: 'Hospital selection required',
        message: 'A hospital must be selected to create documents' 
      }, { status: 400 });
    }
    
    const { 
      title, 
      fileName, 
      fileUrl, 
      fileSize, 
      fileType, 
      status, 
      department, 
      category_id 
    } = await request.json();
    
    // Handle missing category_id by using a default or creating one
    let finalCategoryId = category_id;
    
    if (!finalCategoryId) {
      // Try to find a default category
      const defaultCategory = await prisma.documentCategory.findFirst({
        where: { hospital_id: hospitalId }
      });
      
      if (defaultCategory) {
        finalCategoryId = defaultCategory.id;
      } else {
        // Create a default category if none exists
        const newCategory = await prisma.documentCategory.create({
          data: {
            name: 'Uncategorized',
            icon: 'file-text',
            description: 'Default category for uncategorized documents',
            hospital_id: hospitalId
          }
        });
        
        finalCategoryId = newCategory.id;
      }
    }
    
    const newDocument = await prisma.document.create({
      data: {
        title: title || '',
        fileName: fileName || null,
        fileUrl: fileUrl || null,
        fileSize: fileSize || null,
        fileType: fileType || null,
        status: status || 'active',
        department: department || null,
        updatedBy: user.email,
        category_id: finalCategoryId,
        hospital_id: hospitalId,
      },
      include: {
        category: true,
      },
    });
    
    return NextResponse.json(newDocument, { status: 201 });
  } catch (error) {
    return handleApiError(error);
  }
}

// PUT /api/documentation/documents - Update a document
export async function PUT(request: Request) {
  try {
    const { user, hospitalId, isSuperAdmin } = await getApiContext(request);
    
    const { 
      id,
      title, 
      fileName, 
      fileUrl, 
      fileSize, 
      fileType, 
      status, 
      department, 
      category_id 
    } = await request.json();
    
    if (!id) {
      return NextResponse.json(
        { error: 'Document ID is required' },
        { status: 400 }
      );
    }
    
    // Check if the document exists and user has permission to update it
    const existingDocument = await prisma.document.findUnique({
      where: { id },
    });
    
    if (!existingDocument) {
      return NextResponse.json(
        { error: 'Document not found' },
        { status: 404 }
      );
    }
    
    // Only super_admin or users from the same hospital can update documents
    if (!isSuperAdmin && existingDocument.hospital_id !== hospitalId) {
      return NextResponse.json(
        { error: 'You do not have permission to update this document' },
        { status: 403 }
      );
    }
    
    // Handle missing category_id by using a default or creating one
    let finalCategoryId = category_id;
    
    if (!finalCategoryId && !existingDocument.category_id) {
      // Try to find a default category
      const defaultCategory = await prisma.documentCategory.findFirst({
        where: { hospital_id: hospitalId }
      });
      
      if (defaultCategory) {
        finalCategoryId = defaultCategory.id;
      } else {
        // Create a default category if none exists
        const newCategory = await prisma.documentCategory.create({
          data: {
            name: 'Uncategorized',
            icon: 'file-text',
            description: 'Default category for uncategorized documents',
            hospital_id: hospitalId
          }
        });
        
        finalCategoryId = newCategory.id;
      }
    } else if (!finalCategoryId) {
      finalCategoryId = existingDocument.category_id;
    }
    
    const updatedDocument = await prisma.document.update({
      where: { id },
      data: {
        title: title || existingDocument.title,
        fileName: fileName || existingDocument.fileName,
        fileUrl: fileUrl || existingDocument.fileUrl,
        fileSize: fileSize !== undefined ? fileSize : existingDocument.fileSize,
        fileType: fileType || existingDocument.fileType,
        status: status || existingDocument.status,
        department: department !== undefined ? department : existingDocument.department,
        updatedBy: user.email,
        category_id: finalCategoryId,
        updatedAt: new Date(),
      },
      include: {
        category: true,
      },
    });
    
    return NextResponse.json(updatedDocument);
  } catch (error) {
    return handleApiError(error);
  }
}

// DELETE /api/documentation/documents - Delete a document
export async function DELETE(request: Request) {
  try {
    const { user, hospitalId, isSuperAdmin } = await getApiContext(request);
    
    const url = new URL(request.url);
    const id = url.searchParams.get('id');
    
    if (!id) {
      return NextResponse.json(
        { error: 'Document ID is required' },
        { status: 400 }
      );
    }
    
    // Check if the document exists and user has permission to delete it
    const existingDocument = await prisma.document.findUnique({
      where: { id },
    });
    
    if (!existingDocument) {
      return NextResponse.json(
        { error: 'Document not found' },
        { status: 404 }
      );
    }
    
    // Only super_admin or users from the same hospital can delete documents
    if (!isSuperAdmin && existingDocument.hospital_id !== hospitalId) {
      return NextResponse.json(
        { error: 'You do not have permission to delete this document' },
        { status: 403 }
      );
    }
    
    await prisma.document.delete({
      where: { id },
    });
    
    return NextResponse.json({ success: true });
  } catch (error) {
    return handleApiError(error);
  }
} 