import { NextRequest, NextResponse } from 'next/server';
import { auth } from '@/lib/auth';
import prisma from '@/lib/prisma';
import { getApiContext, handleApiError } from '@/lib/api-helpers';

/**
 * Debug endpoint to check hospital selection across various sources
 */
export async function GET(request: NextRequest) {
  try {
    // Get session and user info
    const session = await auth();
    
    if (!session || !session.user?.email) {
      return NextResponse.json({ 
        status: 'error',
        message: 'Not authenticated'
      }, { status: 401 });
    }
    
    // Get the user from the database
    const user = await prisma.user.findUnique({
      where: { email: session.user.email as string },
      select: { id: true, username: true, email: true, role: true, hospital_id: true }
    });
    
    // Extract hospital ID from request headers
    const headers = new Headers(request.headers);
    const headerHospitalId = headers.get('x-hospital-id');
    
    // Extract hospital ID from URL
    const url = new URL(request.url);
    const urlHospitalId = url.searchParams.get('hospitalId');
    
    // Extract hospital ID from cookies
    const cookieHeader = headers.get('cookie');
    let cookieHospitalId = null;
    if (cookieHeader) {
      const cookies = cookieHeader.split(';').reduce((acc, cookie) => {
        const [name, value] = cookie.trim().split('=').map(c => c.trim());
        if (name && value) acc[name] = value;
        return acc;
      }, {} as Record<string, string>);
      
      cookieHospitalId = cookies.selectedHospitalId || null;
    }
    
    // Get hospital ID from localStorage (client-side only, can't be accessed here)
    
    // Get hospital list
    const hospitals = await prisma.hospital.findMany({
      select: { id: true, name: true }
    });
    
    // Return diagnostic information
    return NextResponse.json({
      status: 'success',
      user: {
        id: user?.id,
        email: user?.email,
        role: user?.role,
        assignedHospitalId: user?.hospital_id
      },
      hospitalSelection: {
        fromHeader: headerHospitalId,
        fromUrl: urlHospitalId,
        fromCookie: cookieHospitalId,
        effective: headerHospitalId || urlHospitalId || cookieHospitalId || (user?.role !== 'super_admin' ? user?.hospital_id : null)
      },
      availableHospitals: hospitals,
      requestInfo: {
        url: request.url,
        method: request.method,
        headers: Array.from(headers.entries()).reduce((acc, [key, value]) => {
          acc[key] = value;
          return acc;
        }, {} as Record<string, string>)
      }
    });
  } catch (error) {
    return handleApiError(error);
  }
} 