import { NextRequest, NextResponse } from 'next/server';
import { auth } from '@/lib/auth';
import prisma from '@/lib/prisma';
import { getApiContext, handleApiError } from '@/lib/api-helpers';

export async function GET(request: NextRequest) {
  try {
    // Use our new helper to get context including hospital ID
    const { user, isSuperAdmin, hospitalId } = await getApiContext(request);
    
    // For debugging
    console.log(`Policies API - Hospital ID: ${hospitalId}, Super Admin: ${isSuperAdmin}`);
    
    // If super admin with no hospital selected, return an error
    if (isSuperAdmin && !hospitalId) {
      return NextResponse.json({ 
        error: 'Hospital selection required',
        message: 'Super admin must select a hospital to view policies data' 
      }, { status: 400 });
    }
    
    // Get compliance policies for the selected/assigned hospital
    const policies = await prisma.compliancePolicy.findMany({
      where: hospitalId ? { hospital_id: hospitalId } : undefined,
      orderBy: [
        { category: 'asc' },
        { name: 'asc' }
      ]
    });
    
    // Group by category
    const groupedPolicies = policies.reduce((acc, policy) => {
      if (!acc[policy.category]) {
        acc[policy.category] = [];
      }
      acc[policy.category].push(policy);
      return acc;
    }, {} as Record<string, typeof policies>);
    
    return NextResponse.json(groupedPolicies);
  } catch (error) {
    return handleApiError(error);
  }
}

export async function POST(req: Request) {
  try {
    // Use our new helper to get context including hospital ID
    const { user, isSuperAdmin, hospitalId } = await getApiContext(req);
    
    // For debugging
    console.log(`Creating policy - Hospital ID: ${hospitalId}, Super Admin: ${isSuperAdmin}`);
    
    // If super admin with no hospital selected, return an error
    if (isSuperAdmin && !hospitalId) {
      return NextResponse.json({ 
        error: 'Hospital selection required',
        message: 'Super admin must select a hospital to add policy data' 
      }, { status: 400 });
    }
    
    // Ensure we have a valid hospital ID
    if (!hospitalId) {
      return NextResponse.json({ 
        error: 'Missing hospital ID',
        message: 'A hospital ID is required to create a policy' 
      }, { status: 400 });
    }
    
    const data = await req.json();
    
    // Validate the data
    if (!data.name || !data.category || !data.status) {
      return NextResponse.json({ 
        error: 'Invalid data. Must include name, category, and status fields.' 
      }, { status: 400 });
    }
    
    // Create the policy for the selected/assigned hospital
    const newPolicy = await prisma.compliancePolicy.create({
      data: {
        name: data.name,
        category: data.category,
        status: data.status,
        hospital_id: hospitalId
      }
    });
    
    return NextResponse.json(newPolicy, { status: 201 });
  } catch (error) {
    return handleApiError(error);
  }
} 