import { NextRequest, NextResponse } from 'next/server';
import { auth } from '@/lib/auth';
import prisma from '@/lib/prisma';
import { hash } from 'bcrypt';

// Helper to generate a secure password
const generateSecurePassword = () => {
  const length = 12;
  const charset = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789!@#$%^&*()_+';
  let password = '';
  for (let i = 0; i < length; i++) {
    const randomIndex = Math.floor(Math.random() * charset.length);
    password += charset[randomIndex];
  }
  return password;
};

// GET - Fetch all hospitals
export async function GET(request: NextRequest) {
  try {
    // Check database connection first
    try {
      await prisma.$connect();
      console.log('Database connection successful');
    } catch (connectionError) {
      console.error('Database connection failed:', 
        connectionError instanceof Error ? connectionError.message : String(connectionError)
      );
      return NextResponse.json({ 
        error: 'Database connection failed', 
        details: connectionError instanceof Error ? connectionError.message : String(connectionError)
      }, { status: 503 });
    }
    
    const session = await auth();
    
    if (!session) {
      return NextResponse.json({ error: 'Unauthorized' }, { status: 401 });
    }
    
    const userEmail = session.user?.email;
    
    // Get user and check if super_admin
    const user = await prisma.user.findUnique({
      where: { email: userEmail as string },
      select: { role: true }
    });
    
    if (!user || user.role !== 'super_admin') {
      return NextResponse.json({ error: 'Forbidden - Requires super admin access' }, { status: 403 });
    }
    
    // Fetch all hospitals with their admin users
    const hospitals = await prisma.hospital.findMany({
      include: {
        users: {
          where: { role: 'hospital' },
          select: { id: true, username: true, email: true }
        }
      },
      orderBy: { name: 'asc' }
    });
    
    return NextResponse.json(hospitals);
  } catch (error) {
    // Fix the console.error null payload issue
    if (error) {
      console.error('Error fetching hospitals:', 
        error instanceof Error ? error.message : String(error)
      );
    }
    
    return NextResponse.json({ 
      error: 'Failed to fetch hospitals',
      details: error instanceof Error ? error.message : String(error)
    }, { status: 500 });
  }
}

// POST - Create a new hospital with admin user
export async function POST(request: NextRequest) {
  try {
    console.log('Starting hospital creation...');
    
    // Test database connection with more specific error handling
    try {
      await prisma.$connect();
      const dbTest = await prisma.hospital.count();
      console.log('Database connection test successful. Hospital count:', dbTest);
    } catch (dbError) {
      // Fix the console.error null payload issue
      if (dbError) {
        console.error('Database connection test failed:', 
          dbError instanceof Error ? dbError.message : String(dbError)
        );
      }
      
      // Check for specific database connection errors
      const errorMessage = dbError instanceof Error ? dbError.message : String(dbError);
      if (errorMessage.includes("Can't reach database server")) {
        return NextResponse.json({ 
          error: 'Database server not reachable', 
          details: 'Please make sure your MySQL server is running at localhost:3006.',
          originalError: errorMessage
        }, { status: 503 });
      }
      
      return NextResponse.json({ 
        error: 'Database connection failed', 
        details: errorMessage
      }, { status: 500 });
    }
    
    // Override NEXTAUTH_URL if needed
    if (process.env.NEXTAUTH_URL !== 'http://localhost:3002') {
      console.log(`Overriding NEXTAUTH_URL from ${process.env.NEXTAUTH_URL} to http://localhost:3002`);
      process.env.NEXTAUTH_URL = 'http://localhost:3002';
    }
    
    const session = await auth();
    
    console.log('Session details:', {
      exists: !!session,
      user: session?.user ? {
        name: session.user.name,
        email: session.user.email,
        role: session.user.role
      } : 'No user in session'
    });
    
    if (!session) {
      console.log('No session found');
      return NextResponse.json({ error: 'Unauthorized' }, { status: 401 });
    }
    
    const userEmail = session.user?.email;
    console.log('User email:', userEmail);
    
    // Get user and check if super_admin
    const user = await prisma.user.findUnique({
      where: { email: userEmail as string },
      select: { role: true }
    });
    
    console.log('User found:', user);
    
    if (!user || user.role !== 'super_admin') {
      console.log('User is not a super admin:', user?.role);
      return NextResponse.json({ error: 'Forbidden - Requires super admin access' }, { status: 403 });
    }
    
    const data = await request.json();
    console.log('Request data:', JSON.stringify(data));
    
    // Validate the hospital data
    if (!data.name || !data.location) {
      console.log('Missing required hospital data');
      return NextResponse.json({ 
        error: 'Invalid data. Hospital name and location are required.' 
      }, { status: 400 });
    }
    
    // Validate admin user data
    if (!data.adminEmail || !data.adminUsername) {
      console.log('Missing required admin user data');
      return NextResponse.json({ 
        error: 'Invalid data. Admin email and username are required.' 
      }, { status: 400 });
    }
    
    // Check if admin email already exists
    const existingUser = await prisma.user.findUnique({
      where: { email: data.adminEmail }
    });
    
    if (existingUser) {
      console.log('Email already exists:', data.adminEmail);
      return NextResponse.json({ 
        error: 'Admin email already exists. Please use a different email.' 
      }, { status: 400 });
    }
    
    console.log('Starting transaction...');
    // Create transaction to ensure both hospital and user are created
    const result = await prisma.$transaction(async (prisma: any) => {
      console.log('Creating hospital...');
      // Create the hospital
      const hospital = await prisma.hospital.create({
        data: {
          name: data.name,
          location: data.location,
          logo_url: data.logoUrl || null
        }
      });
      
      console.log('Hospital created:', hospital);
      
      // Generate a secure password for the admin user
      const password = data.password || generateSecurePassword();
      console.log('Hashing password...');
      const hashedPassword = await hash(password, 10);
      
      console.log('Creating admin user...');
      try {
        // Create the admin user for this hospital
        const adminUser = await prisma.user.create({
          data: {
            username: data.adminUsername,
            email: data.adminEmail,
            password: hashedPassword,
            role: 'hospital',
            hospital_id: hospital.id,
            auth0_id: null // Explicitly set auth0_id to null since it's optional
          }
        });
        
        console.log('Admin user created:', adminUser.id);
        
        return { hospital, adminUser, plainPassword: password };
      } catch (userError) {
        // Fix the console.error null payload issue
        if (userError) {
          console.error('Error creating user:', 
            userError instanceof Error ? userError.message : String(userError)
          );
        }
        
        // Throw a more specific error
        throw new Error(`Failed to create admin user: ${userError instanceof Error ? userError.message : String(userError)}`);
      }
    });
    
    console.log('Transaction completed successfully');
    
    // Fetch the hospital with its users to match the expected format
    const hospitalWithUsers = await prisma.hospital.findUnique({
      where: { id: result.hospital.id },
      include: {
        users: {
          where: { role: 'hospital' },
          select: { id: true, username: true, email: true }
        }
      }
    });
    
    // Return the created hospital and admin info
    return NextResponse.json({
      hospital: hospitalWithUsers,
      adminUser: {
        id: result.adminUser.id,
        username: result.adminUser.username,
        email: result.adminUser.email
      },
      // Include plain password only in response for first-time setup
      temporaryPassword: result.plainPassword
    }, { status: 201 });
  } catch (error) {
    // Fix the console.error null payload issue
    if (error) {
      console.error('Error creating hospital:', 
        error instanceof Error ? error.message : String(error)
      );
    }
    
    // Return more detailed error information for debugging
    return NextResponse.json({ 
      error: 'Failed to create hospital', 
      details: error instanceof Error ? error.message : String(error)
    }, { status: 500 });
  }
} 