'use client'

import React from 'react'
import { Card, CardContent, CardDescription, CardFooter, CardHeader, CardTitle } from '@/components/ui/card'
import { Input } from '@/components/ui/input'
import { Button } from '@/components/ui/button'
import { Label } from '@/components/ui/label'
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs'
import { Separator } from '@/components/ui/separator'
import { useHospital } from '@/lib/context/HospitalContext'
import { useUser } from '@/lib/context/UserContext'
import { toast } from 'sonner'
import { BuildingIcon } from '@/components/ui/icons'
import { Key as KeyIcon, AtSign as AtSignIcon, User as UserIcon } from 'lucide-react'

export default function HospitalProfilePage() {
  const { currentHospital } = useHospital()
  const userContext = useUser()
  
  const [formData, setFormData] = React.useState({
    hospitalName: currentHospital?.name || '',
    hospitalLocation: currentHospital?.location || '',
    userEmail: userContext.isLoggedIn ? userContext.currentUser?.email || '' : '',
    newPassword: ''
  })
  
  const [isLoading, setIsLoading] = React.useState(false)
  
  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const { name, value } = e.target
    setFormData(prev => ({ ...prev, [name]: value }))
  }
  
  const handleProfileUpdate = async (e: React.FormEvent) => {
    e.preventDefault()
    setIsLoading(true)
    
    try {
      // Here you would make an API call to update the hospital profile
      // const response = await fetch('/api/hospital/profile', {
      //   method: 'PUT',
      //   headers: { 'Content-Type': 'application/json' },
      //   body: JSON.stringify({
      //     name: formData.hospitalName,
      //     location: formData.hospitalLocation
      //   })
      // })
      
      // Simulate API call
      await new Promise(resolve => setTimeout(resolve, 1000))
      
      toast.success('Hospital profile updated successfully')
    } catch (error) {
      console.error('Error updating profile:', error)
      toast.error('Failed to update profile')
    } finally {
      setIsLoading(false)
    }
  }
  
  const handlePasswordUpdate = async (e: React.FormEvent) => {
    e.preventDefault()
    
    setIsLoading(true)
    
    try {
      // Here you would make an API call to update the password
      // const response = await fetch('/api/hospital/password', {
      //   method: 'PUT',
      //   headers: { 'Content-Type': 'application/json' },
      //   body: JSON.stringify({
      //     email: formData.userEmail,
      //     newPassword: formData.newPassword
      //   })
      // })
      
      // Simulate API call
      await new Promise(resolve => setTimeout(resolve, 1000))
      
      setFormData(prev => ({
        ...prev,
        newPassword: ''
      }))
      
      toast.success('Password updated successfully')
    } catch (error) {
      console.error('Error updating password:', error)
      toast.error('Failed to update password')
    } finally {
      setIsLoading(false)
    }
  }
  
  if (!currentHospital) {
    return (
      <div className="flex items-center justify-center h-[calc(100vh-200px)]">
        <Card className="w-full max-w-md">
          <CardHeader>
            <CardTitle className="text-center">Hospital Profile</CardTitle>
            <CardDescription className="text-center">No hospital information available</CardDescription>
          </CardHeader>
        </Card>
      </div>
    )
  }
  
  return (
    <div className="container mx-auto py-6 max-w-5xl">
      <div className="flex items-center mb-6">
        <BuildingIcon className="h-8 w-8 mr-3 text-primary" />
        <h1 className="text-3xl font-bold text-gray-900">Hospital Profile</h1>
      </div>
      
      <Tabs defaultValue="profile" className="w-full">
        <TabsList className="mb-6">
          <TabsTrigger value="profile" className="text-base">Profile Information</TabsTrigger>
          <TabsTrigger value="security" className="text-base">Security</TabsTrigger>
        </TabsList>
        
        <TabsContent value="profile">
          <Card>
            <CardHeader>
              <CardTitle>Hospital Information</CardTitle>
              <CardDescription>
                Update your hospital's profile information
              </CardDescription>
            </CardHeader>
            <CardContent>
              <form onSubmit={handleProfileUpdate}>
                <div className="grid gap-6">
                  <div className="grid gap-3">
                    <Label htmlFor="hospitalName" className="flex items-center">
                      <BuildingIcon className="h-4 w-4 mr-2" />
                      Hospital Name
                    </Label>
                    <Input 
                      id="hospitalName" 
                      name="hospitalName"
                      value={formData.hospitalName}
                      onChange={handleInputChange}
                      placeholder="Hospital Name" 
                      required
                    />
                  </div>
                  
                  <div className="grid gap-3">
                    <Label htmlFor="hospitalLocation" className="flex items-center">
                      <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="h-4 w-4 mr-2">
                        <path d="M20 10c0 6-8 12-8 12s-8-6-8-12a8 8 0 0 1 16 0Z"/>
                        <circle cx="12" cy="10" r="3"/>
                      </svg>
                      Location
                    </Label>
                    <Input 
                      id="hospitalLocation" 
                      name="hospitalLocation"
                      value={formData.hospitalLocation}
                      onChange={handleInputChange}
                      placeholder="Hospital Location" 
                      required
                    />
                  </div>
                </div>
                
                <div className="mt-6">
                  <Button type="submit" disabled={isLoading}>
                    {isLoading ? 'Updating...' : 'Update Profile'}
                  </Button>
                </div>
              </form>
            </CardContent>
          </Card>
        </TabsContent>
        
        <TabsContent value="security">
          <Card>
            <CardHeader>
              <CardTitle>Security Settings</CardTitle>
              <CardDescription>
                Update your login credentials
              </CardDescription>
            </CardHeader>
            <CardContent>
              <form onSubmit={handlePasswordUpdate}>
                <div className="grid gap-6">
                  <div className="grid gap-3">
                    <Label htmlFor="userEmail" className="flex items-center">
                      <AtSignIcon className="h-4 w-4 mr-2" />
                      Email Address
                    </Label>
                    <Input 
                      id="userEmail" 
                      name="userEmail"
                      type="email"
                      value={formData.userEmail}
                      onChange={handleInputChange}
                      placeholder="user@hospital.com" 
                      required
                    />
                  </div>
                  
                  <Separator />
                  
                  <div className="grid gap-3">
                    <Label htmlFor="newPassword" className="flex items-center">
                      <KeyIcon className="h-4 w-4 mr-2" />
                      New Password
                    </Label>
                    <Input 
                      id="newPassword" 
                      name="newPassword"
                      type="password"
                      value={formData.newPassword}
                      onChange={handleInputChange}
                      placeholder="••••••••" 
                      required
                    />
                  </div>
                </div>
                
                <div className="mt-6">
                  <Button type="submit" disabled={isLoading}>
                    {isLoading ? 'Updating...' : 'Update Password'}
                  </Button>
                </div>
              </form>
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>
    </div>
  )
} 