'use client'

import { useEffect, useState } from "react"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { useAuditFindings } from "@/hooks/useAuditFindings"
import { format } from "date-fns"
import { toast } from "sonner"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { Textarea } from "@/components/ui/textarea"
import { AlertDialog, AlertDialogAction, AlertDialogCancel, AlertDialogContent, AlertDialogDescription, AlertDialogFooter, AlertDialogHeader, AlertDialogTitle } from "@/components/ui/alert-dialog"
import { Edit, Plus, Trash2, X, Save, XCircle } from "lucide-react"
import { Skeleton } from "@/components/ui/skeleton"
import useDepartments from '@/hooks/useDepartments'
import { useFindingOptions } from '@/hooks/useFindingOptions'

export default function FindingsPage() {
  const { findings, loading, error, refetch, deleteFinding, updateFinding, deleting, updating } = useAuditFindings();
  const { departments, loading: loadingDepts, getDepartmentOptions } = useDepartments();
  const { 
    getSeverityOptions, 
    getStatusOptions, 
    getSeverityColorClass, 
    getStatusColorClass,
    severitiesLoading,
    statusesLoading 
  } = useFindingOptions();
  const [showForm, setShowForm] = useState(false);
  const [activeTab, setActiveTab] = useState('all');
  const [submitting, setSubmitting] = useState(false);
  const [deleteDialogOpen, setDeleteDialogOpen] = useState(false);
  const [findingToDelete, setFindingToDelete] = useState<number | null>(null);
  const [inlineEditingId, setInlineEditingId] = useState<number | null>(null);
  const [formData, setFormData] = useState({
    title: '',
    department: '',
    severity: '',
    status: '',
    description: '',
    resolution_date: '',
    timeline: ''
  });
  const [inlineFormData, setInlineFormData] = useState({
    title: '',
    department: '',
    severity: '',
    status: '',
    description: '',
    resolution_date: '',
    timeline: ''
  });

  // Use static department options as fallback while loading dynamic options
  const departmentOptions = loadingDepts ? [
    { value: 'Nursing', label: 'Nursing' },
    { value: 'Pharmacy', label: 'Pharmacy' },
    { value: 'Laboratory', label: 'Laboratory' }
  ] : getDepartmentOptions();

  useEffect(() => {
    document.title = "Audit Findings | Health Guardian";
  }, []);

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) => {
    const { name, value } = e.target;
    setFormData((prev) => ({ ...prev, [name]: value }));
  };

  const handleInlineInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) => {
    const { name, value } = e.target;
    setInlineFormData((prev) => ({ ...prev, [name]: value }));
  };

  const handleSelectChange = (name: string, value: string) => {
    setFormData((prev) => ({ ...prev, [name]: value }));
  };

  const handleInlineSelectChange = (name: string, value: string) => {
    setInlineFormData((prev) => ({ ...prev, [name]: value }));
  };

  const resetForm = () => {
    setFormData({
      title: '',
      department: '',
      severity: '',
      status: '',
      description: '',
      resolution_date: '',
      timeline: ''
    });
    setShowForm(false);
  };

  const resetInlineForm = () => {
    setInlineFormData({
      title: '',
      department: '',
      severity: '',
      status: '',
      description: '',
      resolution_date: '',
      timeline: ''
    });
    setInlineEditingId(null);
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();

    // Validate inputs
    if (!formData.title || !formData.department || !formData.severity) {
      toast.error("Title, department, and severity are required");
      return;
    }

    setSubmitting(true);

    try {
      if (inlineEditingId) {
        // Update existing finding
        const updateData: any = {
          title: formData.title,
          department: formData.department,
          severity: formData.severity,
          status: formData.status || undefined,
          description: formData.description || null,
        };
        
        // Add resolution date if status is resolved
        if (formData.status === 'resolved') {
          // If resolution date is provided, use it, otherwise use current date
          updateData.resolution_date = formData.resolution_date 
            ? new Date(formData.resolution_date).toISOString() 
            : new Date().toISOString();
        } else {
          // If status is not resolved, set resolution date to null
          updateData.resolution_date = null;
        }

        const success = await updateFinding(inlineEditingId, updateData);
        
        if (success) {
          toast.success("Finding updated successfully");
          resetInlineForm();
        }
      } else {
        // Create new finding
        console.log("Submitting finding data:", {
          title: formData.title,
          department: formData.department,
          severity: formData.severity,
          description: formData.description || null,
          status: 'open',
          timeline: formData.timeline && formData.timeline.trim() !== '' ? formData.timeline : null
        });
        
        try {
          const requestData = {
            title: formData.title,
            department: formData.department,
            severity: formData.severity,
            description: formData.description || null,
            status: 'open',
            timeline: formData.timeline && formData.timeline.trim() !== '' ? formData.timeline : null
          };
          
          console.log("Sending request with data:", requestData);
          
          const response = await fetch("/api/findings", {
            method: "POST",
            headers: {
              "Content-Type": "application/json",
            },
            body: JSON.stringify(requestData),
          });
          
          console.log("Response status:", response.status);
          
          // Try to read the response body as text first
          const responseText = await response.text();
          console.log("Response text:", responseText);
          
          // Then parse it as JSON if possible
          let responseData = null;
          if (responseText) {
            try {
              responseData = JSON.parse(responseText);
            } catch (jsonError) {
              console.error("Failed to parse response as JSON:", jsonError);
              // Continue with responseText if JSON parsing fails
            }
          }
          
          if (!response.ok) {
            console.error("API error response:", responseData || responseText);
            if (responseData) {
              throw new Error(responseData.error || responseData.message || "Failed to create finding");
            } else {
              throw new Error("Server error: " + (responseText || response.statusText));
            }
          }
          
          // If we got here, the request was successful
          console.log("Finding creation successful:", responseData || responseText);
          toast.success(responseData?.message || "Audit finding created successfully");
          resetForm();
          refetch();
                  } catch (fetchError) {
            console.error("Error in fetch operation:", fetchError);
            toast.error(
              fetchError instanceof Error 
                ? fetchError.message 
                : "Failed to communicate with server"
            );
          }
      }
    } catch (err) {
      console.error(inlineEditingId ? "Error updating finding:" : "Error creating finding:", err);
      
                // More detailed error display
          let errorMessage = (err as Error).message;
          
          // Handle common error cases with more user-friendly messages
          if (errorMessage === "Failed to create finding") {
            errorMessage = "Failed to create finding. Please check if you're logged in and have selected a hospital.";
          } else if (errorMessage.includes("Hospital selection required")) {
            errorMessage = "Please select a hospital before adding findings.";
          } else if (errorMessage.includes("Missing hospital")) {
            errorMessage = "Your account is not assigned to a hospital. Please contact an administrator.";
          } else if (errorMessage.includes("Forbidden")) {
            errorMessage = "You don't have permission to perform this action. Regular users can only manage findings for their assigned hospital.";
          }
          
          toast.error(errorMessage, {
            duration: 5000, // Show the error for longer
            id: "findings-error" // Use an ID to prevent duplicate toasts
          });
    } finally {
      setSubmitting(false);
    }
  };

  const handleResolveAction = (id: number) => {
    // Update the finding status to resolved and set resolution date
    updateFinding(id, { 
      status: 'resolved',
      resolution_date: new Date().toISOString()
    })
      .then(success => {
        if (success) {
          toast.success(`Finding #${id} marked as resolved`);
        }
      })
      .catch(error => {
        toast.error("Failed to update status");
      });
  };

  const handleEditFinding = (finding: any) => {
    setInlineFormData({
      title: finding.title,
      department: finding.department,
      severity: finding.severity,
      status: finding.status,
      description: finding.description || '',
      resolution_date: finding.resolution_date ? new Date(finding.resolution_date).toISOString().split('T')[0] : '',
      timeline: finding.timeline ? new Date(finding.timeline).toISOString().split('T')[0] : ''
    });
    setInlineEditingId(finding.id);
  };

  const handleInlineSubmit = async (e: React.FormEvent) => {
    e.preventDefault();

    if (!inlineEditingId) return;

    // Validate inputs
    if (!inlineFormData.title || !inlineFormData.department || !inlineFormData.severity) {
      toast.error("Title, department, and severity are required");
      return;
    }

    setSubmitting(true);

    try {
      const updateData: any = {
        title: inlineFormData.title,
        department: inlineFormData.department,
        severity: inlineFormData.severity,
        status: inlineFormData.status || undefined,
        description: inlineFormData.description || null,
      };
      
      // Add resolution date if status is resolved
      if (inlineFormData.status === 'resolved') {
        updateData.resolution_date = inlineFormData.resolution_date 
          ? new Date(inlineFormData.resolution_date).toISOString() 
          : new Date().toISOString();
      } else {
        updateData.resolution_date = null;
      }

      const success = await updateFinding(inlineEditingId, updateData);
      
      if (success) {
        toast.success("Finding updated successfully");
        resetInlineForm();
      }
    } catch (err) {
      console.error("Error updating finding:", err);
      toast.error((err as Error).message || "Failed to update finding");
    } finally {
      setSubmitting(false);
    }
  };

  const confirmDelete = (id: number) => {
    setFindingToDelete(id);
    setDeleteDialogOpen(true);
  };

  const handleDelete = async () => {
    if (findingToDelete === null) return;
    
    try {
      const success = await deleteFinding(findingToDelete);
      if (success) {
        toast.success("Finding deleted successfully");
      }
    } catch (error) {
      console.error("Error deleting finding:", error);
      toast.error("Failed to delete finding");
    } finally {
      setDeleteDialogOpen(false);
      setFindingToDelete(null);
    }
  };

  const getSeverityBadge = (severity: string) => {
    const severityOption = getSeverityOptions().find(s => s.value === severity);
    if (!severityOption) return null;
    
    // Handle special cases for existing badge variants
    if (severity === 'critical') {
      return <Badge variant="destructive">{severityOption.label}</Badge>;
    } else if (severity === 'low') {
      return <Badge variant="outline" className="text-blue-600">{severityOption.label}</Badge>;
    } else {
      const colorClass = getSeverityColorClass(severity);
      return <Badge className={colorClass}>{severityOption.label}</Badge>;
    }
  };

  const getStatusBadge = (status: string) => {
    const statusOption = getStatusOptions().find(s => s.value === status);
    if (!statusOption) return null;
    
    const colorClass = getStatusColorClass(status);
    return <Badge variant="outline" className={colorClass}>{statusOption.label}</Badge>;
  };

  const filteredFindings = findings.filter(finding => {
    if (activeTab === 'all') return true;
    if (activeTab === 'open') return finding.status === 'open';
    if (activeTab === 'in-progress') return finding.status === 'in_progress';
    if (activeTab === 'resolved') return finding.status === 'resolved';
    return true;
  });

  return (
    <div className="p-6 space-y-6">
      <div className="flex justify-between items-center">
        <div>
          <h1 className="text-2xl font-bold tracking-tight">Audit Findings</h1>
          <p className="text-muted-foreground">Track and manage audit findings across departments</p>
        </div>
        <Button 
          variant="outline" 
          size="sm" 
          onClick={() => {
            if (showForm) {
              setShowForm(false);
            } else if (inlineEditingId) {
              resetInlineForm();
            } else {
              setShowForm(true);
            }
          }}
          className="flex items-center gap-1"
        >
          {showForm ? <X size={16} /> : <Plus size={16} />}
          {showForm ? "Cancel" : "Add Finding"}
        </Button>
      </div>

      {showForm && !inlineEditingId && (
        <Card className="border-dashed border-2 border-blue-200 bg-blue-50">
          <CardHeader>
            <CardTitle>Add New Audit Finding</CardTitle>
            <CardDescription>
              Report a new compliance or regulatory finding
            </CardDescription>
          </CardHeader>
          <CardContent>
            <form onSubmit={handleSubmit} className="space-y-4">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div className="md:col-span-2">
                  <label htmlFor="title" className="block text-sm font-medium mb-1">Finding Title</label>
                  <Input
                    id="title"
                    name="title"
                    placeholder="e.g. Missing documentation for equipment maintenance"
                    value={formData.title}
                    onChange={handleInputChange}
                    className="w-full"
                  />
                </div>
                <div>
                  <label htmlFor="department" className="block text-sm font-medium mb-1">Department</label>
                  <Select value={formData.department} onValueChange={(value) => handleSelectChange('department', value)}>
                    <SelectTrigger className="w-full">
                      <SelectValue placeholder="Select Department" />
                    </SelectTrigger>
                    <SelectContent>
                      {departmentOptions.map(department => (
                        <SelectItem key={department.value} value={department.value}>
                          {department.label}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>
                <div>
                  <label htmlFor="severity" className="block text-sm font-medium mb-1">Severity</label>
                  <Select value={formData.severity} onValueChange={(value) => handleSelectChange('severity', value)}>
                    <SelectTrigger className="w-full">
                      <SelectValue placeholder="Select Severity" />
                    </SelectTrigger>
                    <SelectContent>
                      {getSeverityOptions().map(severity => (
                        <SelectItem key={severity.value} value={severity.value}>
                          {severity.label}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>
                <div>
                  <label htmlFor="timeline" className="block text-sm font-medium mb-1">Timeline for Resolution</label>
                  <Input
                    id="timeline"
                    name="timeline"
                    type="date"
                    value={formData.timeline}
                    onChange={handleInputChange}
                    className="w-full"
                    placeholder="Expected resolution date"
                  />
                </div>
                <div className="md:col-span-2">
                  <label htmlFor="description" className="block text-sm font-medium mb-1">Description (Optional)</label>
                  <Textarea
                    id="description"
                    name="description"
                    placeholder="Provide additional details about the finding"
                    value={formData.description}
                    onChange={handleInputChange}
                    className="w-full"
                    rows={3}
                  />
                </div>
              </div>
              <div className="flex justify-end">
                <Button 
                  type="submit" 
                  disabled={submitting}
                  className="bg-blue-600 hover:bg-blue-700"
                >
                  {submitting ? "Submitting..." : "Submit Finding"}
                </Button>
              </div>
            </form>
          </CardContent>
        </Card>
      )}

      <Tabs defaultValue="all" value={activeTab} onValueChange={setActiveTab} className="w-full">
        <TabsList>
          <TabsTrigger value="all">All Findings</TabsTrigger>
          <TabsTrigger value="open">Open</TabsTrigger>
          <TabsTrigger value="in-progress">In Progress</TabsTrigger>
          <TabsTrigger value="resolved">Resolved</TabsTrigger>
        </TabsList>
        
        <TabsContent value={activeTab} className="space-y-4 mt-4">
          {loading ? (
            <FindingsSkeleton />
          ) : error ? (
            <div className="p-4 text-red-500">Error loading findings: {error}</div>
          ) : filteredFindings.length === 0 ? (
            <div className="text-center py-8 text-gray-500">
              No findings available for this filter
            </div>
          ) : (
            filteredFindings.map((finding) => (
              <Card key={finding.id} className="overflow-hidden">
                <div className="flex">
                  <div className={`w-1 h-full ${getSeverityColorClass(finding.severity)}`}></div>
                  <div className="flex-1">
                    {inlineEditingId === finding.id ? (
                      // Inline Edit Form
                      <div className="p-4 bg-blue-50 border-2 border-blue-200">
                        <div className="flex items-center justify-between mb-4">
                          <h3 className="text-lg font-medium">Edit Finding #{finding.id}</h3>
                          <Button
                            variant="ghost"
                            size="sm"
                            onClick={resetInlineForm}
                            className="text-gray-500 hover:text-gray-700"
                          >
                            <XCircle size={16} />
                          </Button>
                        </div>
                        <form onSubmit={handleInlineSubmit} className="space-y-4">
                          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                            <div className="md:col-span-2">
                              <label className="block text-sm font-medium mb-1">Finding Title</label>
                              <Input
                                name="title"
                                placeholder="Finding title"
                                value={inlineFormData.title}
                                onChange={handleInlineInputChange}
                                className="w-full"
                                required
                              />
                            </div>
                            <div>
                              <label className="block text-sm font-medium mb-1">Department</label>
                              <Select 
                                value={inlineFormData.department} 
                                onValueChange={(value) => handleInlineSelectChange('department', value)}
                              >
                                <SelectTrigger className="w-full">
                                  <SelectValue placeholder="Select Department" />
                                </SelectTrigger>
                                <SelectContent>
                                  {departmentOptions.map(department => (
                                    <SelectItem key={department.value} value={department.value}>
                                      {department.label}
                                    </SelectItem>
                                  ))}
                                </SelectContent>
                              </Select>
                            </div>
                            <div>
                              <label className="block text-sm font-medium mb-1">Severity</label>
                              <Select 
                                value={inlineFormData.severity} 
                                onValueChange={(value) => handleInlineSelectChange('severity', value)}
                              >
                                <SelectTrigger className="w-full">
                                  <SelectValue placeholder="Select Severity" />
                                </SelectTrigger>
                                <SelectContent>
                                  {getSeverityOptions().map(severity => (
                                    <SelectItem key={severity.value} value={severity.value}>
                                      {severity.label}
                                    </SelectItem>
                                  ))}
                                </SelectContent>
                              </Select>
                            </div>
                            <div>
                              <label className="block text-sm font-medium mb-1">Status</label>
                              <Select 
                                value={inlineFormData.status} 
                                onValueChange={(value) => handleInlineSelectChange('status', value)}
                              >
                                <SelectTrigger className="w-full">
                                  <SelectValue placeholder="Select Status" />
                                </SelectTrigger>
                                <SelectContent>
                                  {getStatusOptions().map(status => (
                                    <SelectItem key={status.value} value={status.value}>
                                      {status.label}
                                    </SelectItem>
                                  ))}
                                </SelectContent>
                              </Select>
                            </div>
                            {inlineFormData.status === 'resolved' && (
                              <div>
                                <label className="block text-sm font-medium mb-1">Resolution Date</label>
                                <Input
                                  name="resolution_date"
                                  type="date"
                                  value={inlineFormData.resolution_date}
                                  onChange={handleInlineInputChange}
                                  className="w-full"
                                />
                              </div>
                            )}
                            <div className={inlineFormData.status === 'resolved' ? "md:col-span-2" : "md:col-span-2"}>
                              <label className="block text-sm font-medium mb-1">Description</label>
                              <Textarea
                                name="description"
                                placeholder="Finding description"
                                value={inlineFormData.description}
                                onChange={handleInlineInputChange}
                                className="w-full"
                                rows={3}
                              />
                            </div>
                          </div>
                          <div className="flex justify-end gap-2">
                            <Button 
                              type="button" 
                              variant="outline"
                              onClick={resetInlineForm}
                              disabled={submitting}
                            >
                              Cancel
                            </Button>
                            <Button 
                              type="submit" 
                              disabled={submitting}
                              className="bg-blue-600 hover:bg-blue-700"
                            >
                              {submitting ? (
                                <>
                                  <Save className="h-4 w-4 mr-2 animate-spin" />
                                  Updating...
                                </>
                              ) : (
                                <>
                                  <Save className="h-4 w-4 mr-2" />
                                  Update Finding
                                </>
                              )}
                            </Button>
                          </div>
                        </form>
                      </div>
                    ) : (
                      // Regular Card Content
                      <>
                    <CardHeader className="pb-2">
                      <div className="flex items-start justify-between">
                        <div>
                          <CardTitle className="text-base font-medium">{finding.title}</CardTitle>
                          <CardDescription className="text-xs mt-1">
                            {finding.department} • Finding #{finding.id}
                          </CardDescription>
                        </div>
                        <div className="flex items-center gap-2">
                          {getSeverityBadge(finding.severity)}
                          <button 
                            className="text-gray-400 hover:text-blue-500 rounded-full p-1 hover:bg-gray-100"
                            onClick={() => handleEditFinding(finding)}
                            title="Edit finding"
                          >
                            <Edit size={16} />
                          </button>
                          <button 
                            className="text-gray-400 hover:text-red-500 rounded-full p-1 hover:bg-gray-100"
                            onClick={() => confirmDelete(finding.id)}
                            title="Delete finding"
                          >
                            <Trash2 size={16} />
                          </button>
                        </div>
                      </div>
                    </CardHeader>
                    <CardContent className="pt-0">
                      {finding.description && (
                        <p className="text-sm mb-2 text-gray-700">{finding.description}</p>
                      )}
                      <div className="flex items-center justify-between">
                        <div className="text-sm text-gray-500">
                          <span>Reported: {format(new Date(finding.date_reported), 'MMM dd, yyyy')}</span>
                          {finding.status === 'resolved' && finding.resolution_date && (
                            <span className="ml-4">Resolved: {new Date(finding.resolution_date).toLocaleDateString()}</span>
                          )}
                        </div>
                        <div className="flex items-center gap-2">
                          {getStatusBadge(finding.status)}
                          {finding.status !== 'resolved' && (
                            <button 
                              className="text-xs text-blue-600 hover:text-blue-800"
                              onClick={() => handleResolveAction(finding.id)}
                            >
                              Resolve
                            </button>
                          )}
                        </div>
                      </div>
                    </CardContent>
                      </>
                    )}
                  </div>
                </div>
              </Card>
            ))
          )}
        </TabsContent>
      </Tabs>

      <AlertDialog open={deleteDialogOpen} onOpenChange={setDeleteDialogOpen}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>Delete Finding</AlertDialogTitle>
            <AlertDialogDescription>
              Are you sure you want to delete this finding? This action cannot be undone.
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel disabled={deleting}>Cancel</AlertDialogCancel>
            <AlertDialogAction 
              onClick={handleDelete} 
              disabled={deleting}
              className="bg-red-600 hover:bg-red-700 text-white"
            >
              {deleting ? "Deleting..." : "Delete"}
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </div>
  );
}

const FindingsSkeleton = () => {
  return (
    <>
      {[1, 2, 3].map(i => (
        <Card key={i} className="overflow-hidden">
          <div className="flex">
            <div className="w-1 h-full bg-gray-200"></div>
            <div className="flex-1">
              <CardHeader className="pb-2">
                <div className="flex items-start justify-between">
                  <div>
                    <Skeleton className="h-5 w-64 mb-1" />
                    <Skeleton className="h-3 w-32" />
                  </div>
                  <Skeleton className="h-6 w-16 rounded-full" />
                </div>
              </CardHeader>
              <CardContent className="pt-0">
                <div className="flex items-center justify-between">
                  <Skeleton className="h-4 w-32" />
                  <div className="flex items-center gap-2">
                    <Skeleton className="h-6 w-20 rounded-full" />
                    <Skeleton className="h-4 w-16" />
                  </div>
                </div>
              </CardContent>
            </div>
          </div>
        </Card>
      ))}
    </>
  );
}; 