'use client'

import React, { useState, useEffect } from "react"
import { 
  Card, 
  CardContent, 
  CardDescription, 
  CardHeader, 
  CardTitle 
} from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { 
  Dialog, 
  DialogContent, 
  DialogDescription, 
  DialogFooter, 
  DialogHeader, 
  DialogTitle 
} from "@/components/ui/dialog"
import { Label } from "@/components/ui/label"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { toast } from "sonner"
import { useHospital } from "@/lib/context/HospitalContext"
import { useDocumentOptions } from "@/hooks/useDocumentOptions"
import { 
  FolderOpen, 
  FileText, 
  Download, 
  Search, 
  Trash2, 
  Pencil, 
  Info, 
  Loader2, 
  Plus,
  BarChart,
  Award,
  GraduationCap
} from "lucide-react"

interface DocumentItem {
  id: string;
  title: string;
  fileName?: string;
  fileUrl?: string;
  fileSize?: number;
  fileType?: string;
  status: string;
  department?: string;
  updatedBy: string;
  category_id: string;
  hospital_id?: number;
  createdAt: string;
  updatedAt: string;
  category: {
    id: string;
    name: string;
    icon: string;
    description: string;
  };
}

interface DocumentCategory {
  id: string;
  name: string;
  icon: string;
  description: string;
  count: number;
}

// Icon mapping function for categories
const getCategoryIcon = (iconName: string) => {
  switch(iconName) {
    case 'FileText':
      return <FileText className="h-5 w-5 text-blue-600" />;
    case 'FileCheck':
      return <FileText className="h-5 w-5 text-green-600" />;
    case 'BarChart':
      return <BarChart className="h-5 w-5 text-purple-600" />;
    case 'Award':
      return <Award className="h-5 w-5 text-amber-600" />;
    case 'GraduationCap':
      return <GraduationCap className="h-5 w-5 text-indigo-600" />;
    default:
      return <FileText className="h-5 w-5 text-gray-600" />;
  }
};

// File type icon component
const FileTypeIcon = ({fileName}: {fileName?: string}) => {
  if (!fileName) return <FileText className="h-10 w-10 text-gray-400" />;
  
  const extension = fileName.split('.').pop()?.toLowerCase();
  
  switch(extension) {
    case 'pdf':
      return (
        <div className="flex items-center justify-center w-10 h-10 bg-red-100 rounded">
          <span className="text-xs font-semibold text-red-700">PDF</span>
        </div>
      );
    case 'doc':
    case 'docx':
      return (
        <div className="flex items-center justify-center w-10 h-10 bg-blue-100 rounded">
          <span className="text-xs font-semibold text-blue-700">DOC</span>
        </div>
      );
    case 'xls':
    case 'xlsx':
      return (
        <div className="flex items-center justify-center w-10 h-10 bg-green-100 rounded">
          <span className="text-xs font-semibold text-green-700">XLS</span>
        </div>
      );
    default:
      return (
        <div className="flex items-center justify-center w-10 h-10 bg-gray-100 rounded">
          <span className="text-xs font-semibold text-gray-700">FILE</span>
        </div>
      );
  }
};

export default function DocumentsPage() {
  const { currentHospital } = useHospital();
  const { getStatusOptions, getStatusColorClass, getStatusLabel } = useDocumentOptions();
  
  // Debug hospital context
  console.log('Documents page - Current hospital:', currentHospital);
  
  const [documents, setDocuments] = useState<DocumentItem[]>([]);
  const [categories, setCategories] = useState<DocumentCategory[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [selectedCategory, setSelectedCategory] = useState<string | null>(null);
  const [searchTerm, setSearchTerm] = useState("");
  const [newDocumentDialogOpen, setNewDocumentDialogOpen] = useState(false);
  const [editDocumentDialogOpen, setEditDocumentDialogOpen] = useState(false);
  const [newCategoryDialogOpen, setNewCategoryDialogOpen] = useState(false);
  const [documentToEdit, setDocumentToEdit] = useState<DocumentItem | null>(null);
  const [formData, setFormData] = useState({
    title: '',
    fileName: '',
    status: 'active',
    department: '',
    category_id: ''
  });
  const [categoryFormData, setCategoryFormData] = useState({
    name: '',
    icon: 'FileText',
    description: ''
  });
  // Add file upload state variables
  const [selectedFile, setSelectedFile] = useState<File | null>(null);
  const [uploadProgress, setUploadProgress] = useState(0);
  const [isUploading, setIsUploading] = useState(false);
  
  // Fetch categories and documents
  useEffect(() => {
    console.log('useEffect triggered - Documents page version 2.0');
    console.log('Hospital context in useEffect:', currentHospital);
    
    // Don't fetch if no hospital is selected yet
    if (!currentHospital?.id) {
      console.log('No hospital selected, skipping fetch');
      setLoading(false);
      return;
    }
    
    const fetchData = async () => {
      setLoading(true);
      try {
        console.log('Fetching document categories...');
        // Fetch document categories
        const categoriesResponse = await fetch('/api/documentation/categories');
        console.log('Categories response status:', categoriesResponse.status);
        
        if (!categoriesResponse.ok) {
          const errorText = await categoriesResponse.text();
          console.error('Categories error response:', errorText);
          throw new Error(`Failed to fetch categories: ${categoriesResponse.statusText}`);
        }
        
        const categoriesData = await categoriesResponse.json();
        console.log('Categories data:', categoriesData);
        setCategories(categoriesData);
        
        console.log('Fetching documents...');
        // Check if we have a current hospital
        if (!currentHospital?.id) {
          throw new Error('No hospital selected. Please select a hospital to view documents.');
        }
        
        // Fetch ALL documents for the hospital (no server-side category filtering)
        const documentsUrl = `/api/documentation/documents?hospitalId=${currentHospital.id}&_t=${Date.now()}`;
        
        const documentsResponse = await fetch(documentsUrl);
        console.log('Documents response status:', documentsResponse.status);
        
        if (!documentsResponse.ok) {
          const errorText = await documentsResponse.text();
          console.error('Documents error response:', errorText);
          throw new Error(`Failed to fetch documents: ${documentsResponse.statusText}`);
        }
        
        const documentsData = await documentsResponse.json();
        console.log('Documents data:', documentsData);
        // Log first document structure for debugging
        if (documentsData.length > 0) {
          console.log('First document structure:', {
            id: documentsData[0].id,
            category_id: documentsData[0].category_id,
            categoryIdType: typeof documentsData[0].category_id,
            category: documentsData[0].category
          });
        }
        setDocuments(documentsData);
        
        setError(null);
      } catch (err) {
        console.error('Error fetching data:', err);
        setError(err instanceof Error ? err.message : 'Failed to load documents');
        // Show empty arrays instead of leaving the previous state
        setCategories([]);
        setDocuments([]);
      } finally {
        setLoading(false);
      }
    };
    
    fetchData();
  }, [currentHospital?.id]);
  
  // Handle form input changes
  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) => {
    const { name, value } = e.target;
    setFormData(prev => ({ ...prev, [name]: value }));
  };
  
  // Handle select changes
  const handleSelectChange = (name: string, value: string) => {
    setFormData(prev => ({ ...prev, [name]: value }));
  };
  
  // Handle file selection
  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    if (e.target.files && e.target.files[0]) {
      const file = e.target.files[0];
      setSelectedFile(file);
      setFormData(prev => ({
        ...prev,
        fileName: file.name,
        fileType: file.type,
        fileSize: file.size
      }));
      console.log("File selected:", file.name);
    }
  };

  // Upload the selected file
  const uploadFile = async () => {
    if (!selectedFile) {
      console.log("No file selected for upload");
      return null;
    }
    
    try {
      setIsUploading(true);
      setUploadProgress(0);
      
      const formData = new FormData();
      formData.append('file', selectedFile);
      console.log("Uploading file:", selectedFile.name);
      
      // Simulate progress updates
      const progressInterval = setInterval(() => {
        setUploadProgress(prev => {
          if (prev >= 90) {
            clearInterval(progressInterval);
            return prev;
          }
          return prev + 10;
        });
      }, 300);
      
      const response = await fetch('/api/documentation/upload', {
        method: 'POST',
        body: formData,
      });
      
      clearInterval(progressInterval);
      
      if (!response.ok) {
        throw new Error('Failed to upload file');
      }
      
      setUploadProgress(100);
      const data = await response.json();
      setIsUploading(false);
      console.log("File uploaded successfully:", data);
      
      return data;
    } catch (error) {
      console.error('Error uploading file:', error);
      setIsUploading(false);
      toast.error('Failed to upload file');
      return null;
    }
  };
  
  // Reset form data
  const resetForm = () => {
    setFormData({
      title: '',
      fileName: '',
      status: 'active',
      department: '',
      category_id: ''
    });
    setSelectedFile(null);
    setUploadProgress(0);
    setIsUploading(false);
  };
  
  // Open edit document dialog
  const openEditDialog = (document: DocumentItem) => {
    setDocumentToEdit(document);
    setFormData({
      title: document.title,
      fileName: document.fileName || '',
      status: document.status,
      department: document.department || '',
      category_id: document.category_id
    });
    setEditDocumentDialogOpen(true);
  };
  
  // Create new category
  const createCategory = async () => {
    try {
      if (!categoryFormData.name) {
        toast.error('Category name is required');
        return;
      }
      
      const response = await fetch('/api/documentation/categories', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(categoryFormData),
      });
      
      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.error || 'Failed to create category');
      }
      
      const newCategory = await response.json();
      
      // Update the categories list
      setCategories(prev => [...prev, newCategory]);
      
      toast.success('Category created successfully');
      setCategoryFormData({
        name: '',
        icon: 'FileText',
        description: ''
      });
      setNewCategoryDialogOpen(false);
    } catch (error) {
      console.error('Error creating category:', error);
      toast.error(error instanceof Error ? error.message : 'Failed to create category');
    }
  };
  
  // Create new document
  const createDocument = async () => {
    try {
      let fileData = null;
      
      // Upload file if selected
      if (selectedFile) {
        console.log('Uploading file for new document');
        fileData = await uploadFile();
        if (!fileData) {
          toast.error('Failed to upload file');
          return;
        }
      }
      
      // Prepare document data with file information if available
      const documentData = {
        ...formData,
        ...(fileData && {
          fileName: fileData.fileName || selectedFile?.name,
          fileUrl: fileData.fileUrl,
          fileSize: fileData.fileSize || selectedFile?.size,
          fileType: fileData.fileType || selectedFile?.type
        })
      };
      
      console.log('Sending document data to API:', documentData);
      
      const response = await fetch('/api/documentation/documents', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(documentData),
      });
      
      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.error || 'Failed to create document');
      }
      
      const newDocument = await response.json();
      
      // Update the documents list
      setDocuments(prev => [newDocument, ...prev]);
      
      // Update the category count
      setCategories(prev => 
        prev.map(category => 
          category.id === newDocument.category_id 
            ? { ...category, count: category.count + 1 } 
            : category
        )
      );
      
      toast.success('Document created successfully');
      resetForm();
      setNewDocumentDialogOpen(false);
    } catch (error) {
      console.error('Error creating document:', error);
      toast.error(error instanceof Error ? error.message : 'Failed to create document');
    }
  };
  
  // Update document
  const updateDocument = async () => {
    try {
      if (!documentToEdit) {
        toast.error('No document selected for editing');
        return;
      }
      
      let fileData = null;
      
      // Upload file if selected
      if (selectedFile) {
        console.log('Uploading new file for document update');
        fileData = await uploadFile();
        if (!fileData) {
          toast.error('Failed to upload file');
          return;
        }
      }
      
      // Prepare document data with file information if available
      const documentData = {
        id: documentToEdit.id,
        ...formData,
        // If we have a new file, use its data
        ...(fileData && {
          fileName: fileData.fileName || selectedFile?.name,
          fileUrl: fileData.fileUrl,
          fileSize: fileData.fileSize || selectedFile?.size,
          fileType: fileData.fileType || selectedFile?.type
        }),
        // If no new file and document already has a file, preserve existing file data
        ...(!fileData && documentToEdit.fileName && {
          fileName: documentToEdit.fileName,
          fileUrl: documentToEdit.fileUrl,
          fileSize: documentToEdit.fileSize,
          fileType: documentToEdit.fileType
        })
      };
      
      console.log('Sending updated document data to API:', documentData);
      
      const response = await fetch('/api/documentation/documents', {
        method: 'PUT',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(documentData),
      });
      
      if (!response.ok) {
        throw new Error('Failed to update document');
      }
      
      const updatedDocument = await response.json();
      
      // Update the documents list
      setDocuments(prev => 
        prev.map(doc => 
          doc.id === updatedDocument.id ? updatedDocument : doc
        )
      );
      
      // If category changed, update category counts
      if (documentToEdit.category_id !== updatedDocument.category_id) {
        setCategories(prev => 
          prev.map(category => {
            if (category.id === documentToEdit.category_id) {
              return { ...category, count: category.count - 1 };
            }
            if (category.id === updatedDocument.category_id) {
              return { ...category, count: category.count + 1 };
            }
            return category;
          })
        );
      }
      
      toast.success('Document updated successfully');
      setDocumentToEdit(null);
      resetForm();
      setEditDocumentDialogOpen(false);
    } catch (error) {
      console.error('Error updating document:', error);
      toast.error('Failed to update document');
    }
  };
  
  // Delete document
  const deleteDocument = async (id: string, categoryId: string) => {
    if (!confirm('Are you sure you want to delete this document?')) {
      return;
    }
    
    try {
      const response = await fetch(`/api/documentation/documents?id=${id}`, {
        method: 'DELETE',
      });
      
      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.error || 'Failed to delete document');
      }
      
      // Remove document from list
      setDocuments(prev => prev.filter(doc => doc.id !== id));
      
      // Update category count
      setCategories(prev => 
        prev.map(category => 
          category.id === categoryId 
            ? { ...category, count: category.count - 1 } 
            : category
        )
      );
      
      toast.success('Document deleted successfully');
    } catch (error) {
      console.error('Error deleting document:', error);
      toast.error(error instanceof Error ? error.message : 'Failed to delete document');
    }
  };
  
  // Download document
  const downloadDocument = (doc: DocumentItem) => {
    try {
      if (!doc.fileUrl) {
        toast.error('No file available for download');
        return;
      }
      
      // Create an anchor element and trigger download
      const anchor = document.createElement('a');
      anchor.href = doc.fileUrl;
      anchor.download = doc.fileName || 'document';
      anchor.target = '_blank';
      document.body.appendChild(anchor);
      anchor.click();
      document.body.removeChild(anchor);
    } catch (error) {
      console.error('Error downloading document:', error);
      toast.error('Failed to download document');
    }
  };
  
  // Format date for display
  const formatDate = (dateString: string) => {
    const date = new Date(dateString);
    return date.toLocaleDateString('en-US', { 
      year: 'numeric', 
      month: 'short', 
      day: 'numeric' 
    });
  };

  // Filter documents based on search term and selected category
  const filteredDocuments = documents.filter(doc => {
    const matchesSearch = doc.title.toLowerCase().includes(searchTerm.toLowerCase()) ||
      (doc.department && doc.department.toLowerCase().includes(searchTerm.toLowerCase())) ||
      doc.category.name.toLowerCase().includes(searchTerm.toLowerCase());
    
    const matchesCategory = selectedCategory ? doc.category_id === selectedCategory : true;
    
    // Temporary debug logging to diagnose the issue
    if (selectedCategory && documents.length > 0) {
      console.log('Debug info:', {
        selectedCategory,
        docCategoryId: doc.category_id,
        docTitle: doc.title,
        typeOfSelectedCategory: typeof selectedCategory,
        typeOfDocCategoryId: typeof doc.category_id,
        matchesCategory,
        areEqual: doc.category_id === selectedCategory,
        areEqualLoose: doc.category_id == selectedCategory
      });
    }
    
    return matchesSearch && matchesCategory;
  });

  if (loading) {
    return (
      <div className="flex items-center justify-center h-[calc(100vh-200px)]">
        <div className="flex flex-col items-center">
          <Loader2 className="h-8 w-8 text-blue-500 animate-spin mb-2" />
          <p className="text-sm text-gray-500">Loading documents...</p>
        </div>
      </div>
    );
  }

  if (!currentHospital?.id) {
    return (
      <Card>
        <CardContent className="flex flex-col items-center justify-center py-10 text-center">
          <Info className="h-10 w-10 text-blue-300 mb-3" />
          <h3 className="font-medium mb-1">No Hospital Selected</h3>
          <p className="text-sm text-muted-foreground max-w-md">
            Please select a hospital to view documents. You can do this from the hospital selector in the navigation.
          </p>
        </CardContent>
      </Card>
    );
  }

  if (error) {
    return (
      <Card>
        <CardContent className="flex flex-col items-center justify-center py-10 text-center">
          <Info className="h-10 w-10 text-red-300 mb-3" />
          <h3 className="font-medium mb-1">Error loading documents</h3>
          <p className="text-sm text-muted-foreground max-w-md">
            {error}
          </p>
        </CardContent>
      </Card>
    );
  }

  return (
    <div className="space-y-6">
      <Card className="mb-4 bg-white shadow-sm border-blue-100">
        <CardHeader>
          <CardTitle className="text-lg">Documents Folder - {currentHospital?.name}</CardTitle>
          <CardDescription>
            Browse and manage document files
          </CardDescription>
        </CardHeader>
        {process.env.NODE_ENV === 'development' && (
          <div className="px-6 pb-4">
            <Button variant="outline" size="sm" onClick={() => window.location.href = '/direct-login'}>
              Development: Direct Login
            </Button>
          </div>
        )}
      </Card>
      
      <div className="flex flex-col md:flex-row gap-4">
        {/* Sidebar with categories */}
        <div className="w-full md:w-64 space-y-3">
          <div className="flex items-center justify-between">
            <div className="font-medium text-sm">Categories</div>
            <Button 
              variant="outline" 
              size="sm" 
              className="h-8 text-xs"
              onClick={() => setNewCategoryDialogOpen(true)}
            >
              <FolderOpen className="h-3 w-3 mr-1" />
              New Category
            </Button>
          </div>
          
          <Card className="p-1">
            <div className="space-y-1">
              <Button
                variant="ghost"
                className={`w-full justify-start font-normal rounded-sm ${!selectedCategory ? 'bg-blue-50 text-blue-700' : ''}`}
                onClick={() => setSelectedCategory(null)}
              >
                <FolderOpen className="mr-2 h-5 w-5" />
                All Documents
              </Button>
              
              {categories.map(category => (
                <Button
                  key={category.id}
                  variant="ghost"
                  className={`w-full justify-start font-normal rounded-sm ${selectedCategory === category.id ? 'bg-blue-50 text-blue-700' : ''}`}
                  onClick={() => {
                    console.log('Category clicked:', {
                      categoryId: category.id,
                      categoryName: category.name,
                      typeOfId: typeof category.id
                    });
                    setSelectedCategory(category.id);
                  }}
                >
                  {getCategoryIcon(category.icon)}
                  <span className="ml-2 truncate">{category.name}</span>
                  <Badge className="ml-auto">{category.count}</Badge>
                </Button>
              ))}
            </div>
          </Card>
        </div>
        
        {/* Main content area */}
        <div className="flex-1">
          <div className="flex items-center justify-between mb-4">
            <div className="text-sm font-medium">
              {selectedCategory 
                ? categories.find(c => c.id === selectedCategory)?.name + " Documents" 
                : "All Documents"} ({filteredDocuments.length})
            </div>
            
            <div className="flex items-center space-x-2">
              <div className="relative">
                <div className="absolute inset-y-0 left-0 flex items-center pl-3 pointer-events-none">
                  <Search className="h-4 w-4 text-gray-400" />
                </div>
                <Input
                  type="search"
                  className="pl-10 pr-4 py-2 w-64"
                  placeholder="Search files..."
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                />
              </div>
              <Button 
                onClick={() => setNewDocumentDialogOpen(true)}
                size="sm"
                className="h-9"
              >
                <Plus className="h-4 w-4 mr-1" />
                Add Document
              </Button>
            </div>
          </div>
          
          <Card>
            <CardContent className="p-0">
              {filteredDocuments.length > 0 ? (
                <table className="w-full">
                  <thead className="bg-gray-50 text-xs font-medium text-gray-500 uppercase tracking-wider">
                    <tr>
                      <th className="text-left p-3">Name</th>
                      <th className="text-left p-3 hidden md:table-cell">Category</th>
                      <th className="text-left p-3 hidden md:table-cell">Status</th>
                      <th className="text-left p-3 hidden md:table-cell">Modified</th>
                      <th className="text-left p-3 hidden md:table-cell">Department</th>
                      <th className="text-right p-3">Actions</th>
                    </tr>
                  </thead>
                  <tbody className="divide-y divide-gray-100">
                    {filteredDocuments.map(doc => (
                      <tr key={doc.id} className="hover:bg-gray-50">
                        <td className="p-3">
                          <div className="flex items-center space-x-3">
                            <FileTypeIcon fileName={doc.fileName} />
                            <div>
                              <div className="font-medium">{doc.title}</div>
                              <div className="text-xs text-gray-500">{doc.fileName}</div>
                            </div>
                          </div>
                        </td>
                        <td className="p-3 hidden md:table-cell">
                          <Badge className="font-normal" variant="outline">
                            {doc.category.name}
                          </Badge>
                        </td>
                        <td className="p-3 hidden md:table-cell">
                          <Badge variant="outline" className={getStatusColorClass(doc.status)}>
                            {getStatusLabel(doc.status)}
                          </Badge>
                        </td>
                        <td className="p-3 text-sm text-gray-500 hidden md:table-cell">
                          {formatDate(doc.updatedAt)}
                        </td>
                        <td className="p-3 text-sm text-gray-500 hidden md:table-cell">
                          {doc.department || '-'}
                        </td>
                        <td className="p-3 text-right">
                          <div className="flex justify-end space-x-2">
                            <Button
                              variant="ghost"
                              size="icon"
                              onClick={() => openEditDialog(doc)}
                            >
                              <Pencil className="h-4 w-4" />
                            </Button>
                            <Button
                              variant="ghost"
                              size="icon"
                              onClick={() => downloadDocument(doc)}
                            >
                              <Download className="h-4 w-4" />
                            </Button>
                            <Button
                              variant="ghost"
                              size="icon"
                              onClick={() => deleteDocument(doc.id, doc.category_id)}
                            >
                              <Trash2 className="h-4 w-4 text-red-500" />
                            </Button>
                          </div>
                        </td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              ) : (
                <div className="flex flex-col items-center justify-center py-12 text-center">
                  <Info className="h-10 w-10 text-gray-300 mb-3" />
                  {searchTerm ? (
                    <>
                      <h3 className="font-medium mb-1">No documents match your search</h3>
                      <p className="text-sm text-gray-500 mb-4">
                        Try adjusting your search terms or clear the search
                      </p>
                      <Button variant="outline" onClick={() => setSearchTerm("")}>
                        Clear Search
                      </Button>
                    </>
                  ) : (
                    <>
                      <h3 className="font-medium mb-1">No documents found</h3>
                      <p className="text-sm text-gray-500 mb-4">
                        Get started by adding your first document
                      </p>
                      <Button onClick={() => setNewDocumentDialogOpen(true)}>
                        Add Document
                      </Button>
                    </>
                  )}
                </div>
              )}
            </CardContent>
          </Card>
        </div>
      </div>
      
      {/* New Document Dialog */}
      <Dialog open={newDocumentDialogOpen} onOpenChange={setNewDocumentDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Add New Document</DialogTitle>
            <DialogDescription>
              Create a new document for your hospital.
            </DialogDescription>
          </DialogHeader>
          
          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <Label htmlFor="title">Document Title</Label>
              <Input 
                id="title" 
                name="title" 
                placeholder="Enter document title" 
                value={formData.title}
                onChange={handleInputChange}
              />
            </div>
            
            <div className="space-y-2">
              <Label htmlFor="file">Upload Document</Label>
              <div className="grid w-full max-w-sm items-center gap-1.5">
                <Input
                  id="file"
                  type="file"
                  onChange={handleFileChange}
                  className="cursor-pointer border-2 border-dashed border-gray-300 p-4"
                />
                {isUploading && (
                  <div className="w-full h-2 bg-gray-200 rounded-full mt-2">
                    <div 
                      className="h-full bg-blue-600 rounded-full" 
                      style={{ width: `${uploadProgress}%` }}
                    />
                  </div>
                )}
                {selectedFile && (
                  <p className="text-sm text-green-600 font-medium mt-1">
                    File selected: {selectedFile.name} ({Math.round(selectedFile.size / 1024)} KB)
                  </p>
                )}
                {!selectedFile && (
                  <p className="text-xs text-muted-foreground mt-1">
                    Select a file to upload (PDF, Word, Excel, etc.)
                  </p>
                )}
              </div>
            </div>
            
            <div className="space-y-2">
              <Label htmlFor="department">Department</Label>
              <Input 
                id="department" 
                name="department" 
                placeholder="Clinical Operations" 
                value={formData.department}
                onChange={handleInputChange}
              />
            </div>
            
            <div className="space-y-2">
              <Label htmlFor="category">Category</Label>
              <Select 
                value={formData.category_id} 
                onValueChange={(value) => handleSelectChange('category_id', value)}
              >
                <SelectTrigger>
                  <SelectValue placeholder="Select a category" />
                </SelectTrigger>
                <SelectContent>
                  {categories.map(category => (
                    <SelectItem key={category.id} value={category.id}>
                      {category.name}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            
            <div className="space-y-2">
              <Label htmlFor="status">Status</Label>
              <Select 
                value={formData.status} 
                onValueChange={(value) => handleSelectChange('status', value)}
              >
                <SelectTrigger>
                  <SelectValue placeholder="Select a status" />
                </SelectTrigger>
                <SelectContent>
                  {getStatusOptions().map(option => (
                    <SelectItem key={option.value} value={option.value}>
                      {option.label}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
          </div>
          
          <DialogFooter>
            <Button variant="outline" onClick={() => setNewDocumentDialogOpen(false)}>
              Cancel
            </Button>
            <Button onClick={createDocument}>
              Create Document
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
      
      {/* Edit Document Dialog */}
      <Dialog open={editDocumentDialogOpen} onOpenChange={setEditDocumentDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Edit Document</DialogTitle>
            <DialogDescription>
              Update the document information.
            </DialogDescription>
          </DialogHeader>
          
          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <Label htmlFor="edit-title">Document Title</Label>
              <Input 
                id="edit-title" 
                name="title" 
                placeholder="Enter document title" 
                value={formData.title}
                onChange={handleInputChange}
              />
            </div>
            
            <div className="space-y-2">
              <Label htmlFor="edit-file">Upload Document</Label>
              <div className="grid w-full max-w-sm items-center gap-1.5">
                <Input
                  id="edit-file"
                  type="file"
                  onChange={handleFileChange}
                  className="cursor-pointer border-2 border-dashed border-gray-300 p-4"
                />
                {isUploading && (
                  <div className="w-full h-2 bg-gray-200 rounded-full mt-2">
                    <div 
                      className="h-full bg-blue-600 rounded-full" 
                      style={{ width: `${uploadProgress}%` }}
                    />
                  </div>
                )}
                {selectedFile && (
                  <p className="text-sm text-green-600 font-medium mt-1">
                    File selected: {selectedFile.name} ({Math.round(selectedFile.size / 1024)} KB)
                  </p>
                )}
                {!selectedFile && documentToEdit?.fileName && (
                  <p className="text-sm text-blue-600 mt-1">
                    Current file: {documentToEdit.fileName}
                  </p>
                )}
                {!selectedFile && !documentToEdit?.fileName && (
                  <p className="text-xs text-muted-foreground mt-1">
                    Select a file to upload (PDF, Word, Excel, etc.)
                  </p>
                )}
              </div>
            </div>
            
            <div className="space-y-2">
              <Label htmlFor="edit-department">Department</Label>
              <Input 
                id="edit-department" 
                name="department" 
                placeholder="Clinical Operations" 
                value={formData.department}
                onChange={handleInputChange}
              />
            </div>
            
            <div className="space-y-2">
              <Label htmlFor="edit-category">Category</Label>
              <Select 
                value={formData.category_id} 
                onValueChange={(value) => handleSelectChange('category_id', value)}
              >
                <SelectTrigger>
                  <SelectValue placeholder="Select a category" />
                </SelectTrigger>
                <SelectContent>
                  {categories.map(category => (
                    <SelectItem key={category.id} value={category.id}>
                      {category.name}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            
            <div className="space-y-2">
              <Label htmlFor="edit-status">Status</Label>
              <Select 
                value={formData.status} 
                onValueChange={(value) => handleSelectChange('status', value)}
              >
                <SelectTrigger>
                  <SelectValue placeholder="Select a status" />
                </SelectTrigger>
                <SelectContent>
                  {getStatusOptions().map(option => (
                    <SelectItem key={option.value} value={option.value}>
                      {option.label}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
          </div>
          
          <DialogFooter>
            <Button variant="outline" onClick={() => setEditDocumentDialogOpen(false)}>
              Cancel
            </Button>
            <Button onClick={updateDocument}>
              Update Document
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
      
      {/* New Category Dialog */}
      <Dialog open={newCategoryDialogOpen} onOpenChange={setNewCategoryDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Add New Category</DialogTitle>
            <DialogDescription>
              Create a new category for organizing documents.
            </DialogDescription>
          </DialogHeader>
          
          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <Label htmlFor="name">Category Name</Label>
              <Input 
                id="name" 
                name="name" 
                placeholder="Enter category name" 
                value={categoryFormData.name}
                onChange={(e) => setCategoryFormData({...categoryFormData, name: e.target.value})}
              />
            </div>
            
            <div className="space-y-2">
              <Label htmlFor="icon">Icon</Label>
              <Select 
                value={categoryFormData.icon} 
                onValueChange={(value) => setCategoryFormData({...categoryFormData, icon: value})}
              >
                <SelectTrigger>
                  <SelectValue placeholder="Select an icon" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="FileText">Document</SelectItem>
                  <SelectItem value="FileCheck">Checklist</SelectItem>
                  <SelectItem value="BarChart">Reports</SelectItem>
                  <SelectItem value="Award">Certifications</SelectItem>
                  <SelectItem value="GraduationCap">Training</SelectItem>
                </SelectContent>
              </Select>
            </div>
            
            <div className="space-y-2">
              <Label htmlFor="description">Description</Label>
              <Input 
                id="description" 
                name="description" 
                placeholder="Enter category description" 
                value={categoryFormData.description}
                onChange={(e) => setCategoryFormData({...categoryFormData, description: e.target.value})}
              />
            </div>
          </div>
          
          <DialogFooter>
            <Button variant="outline" onClick={() => setNewCategoryDialogOpen(false)}>
              Cancel
            </Button>
            <Button onClick={createCategory}>
              Create Category
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
} 