'use client'

import React, { useEffect, useState } from 'react'
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card'
import { Button } from '@/components/ui/button'
import { useSelectedHospital } from '@/hooks/useSelectedHospital'
import { useHospital } from '@/lib/context/HospitalContext'
import { useUser } from '@/lib/context/UserContext'

interface DebugInfo {
  user: {
    id: string;
    email: string;
    role: string;
    assignedHospitalId: number | null;
  };
  hospitalSelection: {
    fromHeader: string | null;
    fromUrl: string | null;
    fromCookie: string | null;
    effective: string | number | null;
  };
  availableHospitals: Array<{
    id: number;
    name: string;
  }>;
  requestInfo: {
    url: string;
    method: string;
    headers: Record<string, string>;
  };
}

export default function HospitalSelectionDebugPage() {
  const [debugInfo, setDebugInfo] = useState<DebugInfo | null>(null);
  const [loading, setLoading] = useState(false);
  const { currentHospital, accessibleHospitals } = useHospital();
  const { isSuperAdmin } = useUser();
  const { getHospitalUrlParam, getHospitalFetchOptions } = useSelectedHospital();

  const fetchDebugInfo = async () => {
    setLoading(true);
    try {
      const urlParam = getHospitalUrlParam();
      const options = getHospitalFetchOptions();
      
      const response = await fetch(`/api/debug/hospital-selection${urlParam}`, options);
      const data = await response.json();
      setDebugInfo(data);
    } catch (error) {
      console.error('Error fetching debug info:', error);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchDebugInfo();
  }, []);

  const pageLoadTime = new Date().toLocaleTimeString();

  return (
    <div className="space-y-6">
      <div className="flex justify-between items-center">
        <h1 className="text-2xl font-bold">Hospital Selection Debug</h1>
        <Button onClick={fetchDebugInfo} disabled={loading}>
          {loading ? 'Loading...' : 'Refresh Debug Info'}
        </Button>
      </div>

      <div className="grid gap-6 md:grid-cols-2">
        <Card>
          <CardHeader>
            <CardTitle>Current Hospital Context</CardTitle>
            <CardDescription>Information from the HospitalContext</CardDescription>
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              <div>
                <h3 className="font-medium mb-1">Current Hospital:</h3>
                {currentHospital ? (
                  <div className="bg-green-50 p-3 rounded border border-green-200">
                    <p><strong>ID:</strong> {currentHospital.id}</p>
                    <p><strong>Name:</strong> {currentHospital.name}</p>
                    <p><strong>Location:</strong> {currentHospital.location}</p>
                  </div>
                ) : (
                  <p className="text-orange-500">No hospital selected</p>
                )}
              </div>

              <div>
                <h3 className="font-medium mb-1">LocalStorage:</h3>
                <div className="bg-gray-50 p-3 rounded border">
                  {typeof window !== 'undefined' && (
                    <p>
                      <strong>selectedHospitalId:</strong> {localStorage.getItem('selectedHospitalId') || 'Not set'}
                    </p>
                  )}
                </div>
              </div>

              <div>
                <h3 className="font-medium mb-1">Cookie:</h3>
                <div className="bg-gray-50 p-3 rounded border">
                  <p>
                    <strong>selectedHospitalId:</strong> {document.cookie.split(';').find(c => c.trim().startsWith('selectedHospitalId='))?.split('=')[1] || 'Not set'}
                  </p>
                </div>
              </div>

              <div>
                <h3 className="font-medium mb-1">Super Admin:</h3>
                <p>{isSuperAdmin ? '✅ Yes' : '❌ No'}</p>
              </div>

              <div>
                <h3 className="font-medium mb-1">Accessible Hospitals:</h3>
                <ul className="list-disc pl-5">
                  {accessibleHospitals.map(hospital => (
                    <li key={hospital.id}>
                      {hospital.name} (ID: {hospital.id})
                      {currentHospital?.id === hospital.id && ' ✓'}
                    </li>
                  ))}
                </ul>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle>Server-Side Detection</CardTitle>
            <CardDescription>How the server sees your hospital selection</CardDescription>
          </CardHeader>
          <CardContent>
            {debugInfo ? (
              <div className="space-y-4">
                <div>
                  <h3 className="font-medium mb-1">User:</h3>
                  <div className="bg-gray-50 p-3 rounded border">
                    <p><strong>ID:</strong> {debugInfo.user.id}</p>
                    <p><strong>Email:</strong> {debugInfo.user.email}</p>
                    <p><strong>Role:</strong> {debugInfo.user.role}</p>
                    <p><strong>Assigned Hospital ID:</strong> {debugInfo.user.assignedHospitalId ?? 'None'}</p>
                  </div>
                </div>

                <div>
                  <h3 className="font-medium mb-1">Hospital Selection Sources:</h3>
                  <div className="bg-gray-50 p-3 rounded border">
                    <p><strong>From Header:</strong> {debugInfo.hospitalSelection.fromHeader ?? 'Not set'}</p>
                    <p><strong>From URL:</strong> {debugInfo.hospitalSelection.fromUrl ?? 'Not set'}</p>
                    <p><strong>From Cookie:</strong> {debugInfo.hospitalSelection.fromCookie ?? 'Not set'}</p>
                    <p className="mt-2 font-medium">
                      <strong>Effective Hospital ID:</strong> {debugInfo.hospitalSelection.effective ?? 'None'}
                    </p>
                  </div>
                </div>

                <div>
                  <h3 className="font-medium mb-1">Request Information:</h3>
                  <div className="bg-gray-50 p-3 rounded border text-xs">
                    <p><strong>URL:</strong> {debugInfo.requestInfo.url}</p>
                    <p><strong>Method:</strong> {debugInfo.requestInfo.method}</p>
                    <p><strong>Headers:</strong></p>
                    <div className="ml-4 mt-1">
                      {Object.entries(debugInfo.requestInfo.headers)
                        .filter(([key]) => !key.includes('cookie'))
                        .map(([key, value]) => (
                          <p key={key}><strong>{key}:</strong> {value}</p>
                        ))
                      }
                    </div>
                  </div>
                </div>
              </div>
            ) : (
              <div className="py-8 text-center text-gray-500">
                {loading ? 'Loading debug information...' : 'Failed to load debug information.'}
              </div>
            )}
          </CardContent>
        </Card>
      </div>

      <Card>
        <CardHeader>
          <CardTitle>Hospital Data Filtering Test</CardTitle>
          <CardDescription>Test API endpoints with hospital filtering</CardDescription>
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-2 gap-4">
            <Button 
              variant="outline" 
              onClick={() => window.open(`/api/findings${getHospitalUrlParam()}`, '_blank')}
            >
              Test Findings API
            </Button>
            <Button 
              variant="outline" 
              onClick={() => window.open(`/api/compliance${getHospitalUrlParam()}`, '_blank')}
            >
              Test Compliance API
            </Button>
            <Button 
              variant="outline" 
              onClick={() => window.open(`/api/compliance/standards${getHospitalUrlParam()}`, '_blank')}
            >
              Test Standards API
            </Button>
            <Button 
              variant="outline" 
              onClick={() => window.open(`/api/compliance/policies${getHospitalUrlParam()}`, '_blank')}
            >
              Test Policies API
            </Button>
          </div>
        </CardContent>
      </Card>

      <Card>
        <CardHeader>
          <CardTitle>Page Refresh Information</CardTitle>
          <CardDescription>When the hospital changes, the page will automatically refresh</CardDescription>
        </CardHeader>
        <CardContent>
          <div className="space-y-2">
            <p><strong>Page loaded at:</strong> {pageLoadTime}</p>
            <p className="text-sm text-muted-foreground">
              When you change hospitals using the selector in the navbar, the page will 
              automatically refresh to load data for the selected hospital.
            </p>
            <div className="bg-amber-50 border border-amber-200 rounded p-3 mt-2">
              <p className="text-amber-800 text-sm">
                <strong>Note:</strong> If you're viewing this page and change hospitals, 
                you'll be redirected back to this page after refresh.
              </p>
            </div>
          </div>
        </CardContent>
      </Card>
    </div>
  )
} 