'use client';

import { useEffect, useState } from 'react';
import { useRouter } from 'next/navigation';

export default function AccessTestPage() {
  const [user, setUser] = useState<any>(null);
  const [error, setError] = useState('');
  const [isLoading, setIsLoading] = useState(true);
  const router = useRouter();

  // Check for our custom cookie
  useEffect(() => {
    const checkSession = async () => {
      try {
        // Try to read the cookie (this will only work client-side)
        const cookies = document.cookie.split(';').reduce((acc, cookie) => {
          const [key, value] = cookie.trim().split('=');
          try {
            acc[key] = decodeURIComponent(value);
          } catch (e) {
            acc[key] = value;
          }
          return acc;
        }, {} as Record<string, string>);

        if (cookies['user-session']) {
          try {
            // Parse the user data
            const userData = JSON.parse(cookies['user-session']);
            setUser(userData);
            setError('');
          } catch (e) {
            setError('Invalid session format');
          }
        } else {
          setError('No session found');
        }
      } catch (e) {
        setError(`Error checking session: ${e instanceof Error ? e.message : 'Unknown error'}`);
      } finally {
        setIsLoading(false);
      }
    };

    checkSession();
  }, []);

  const handleLogout = () => {
    // Clear the cookie
    document.cookie = 'user-session=; Path=/; Expires=Thu, 01 Jan 1970 00:00:01 GMT;';
    // Redirect to login
    router.push('/login');
  };

  if (isLoading) {
    return (
      <div className="flex items-center justify-center min-h-screen">
        <p>Loading...</p>
      </div>
    );
  }

  return (
    <div className="p-8">
      <h1 className="text-2xl font-bold mb-6">Protected Page Access Test</h1>
      
      {error ? (
        <div className="bg-red-50 p-4 rounded-md text-red-700 mb-4">
          <p className="font-bold">Error</p>
          <p>{error}</p>
          <button
            onClick={() => router.push('/direct-login')}
            className="mt-4 px-4 py-2 bg-blue-600 text-white rounded-md"
          >
            Go to Direct Login
          </button>
        </div>
      ) : (
        <div className="bg-green-50 p-4 rounded-md text-green-700 mb-4">
          <p className="font-bold">Success! You are logged in</p>
          <pre className="mt-2 p-2 bg-green-100 rounded overflow-auto">
            {JSON.stringify(user, null, 2)}
          </pre>
          <button 
            onClick={handleLogout}
            className="mt-4 px-4 py-2 bg-blue-600 text-white rounded-md"
          >
            Logout
          </button>
        </div>
      )}
    </div>
  );
} 