'use client'

import { useState, useEffect } from 'react'
import { signIn, useSession } from 'next-auth/react'
import { useRouter } from 'next/navigation'

// Simple toast service
const toast = {
  success: (title: string, options?: { description?: string }) => {
    console.log('SUCCESS:', title, options?.description)
    alert(`${title}\n${options?.description || ''}`)
  },
  error: (title: string, options?: { description?: string }) => {
    console.log('ERROR:', title, options?.description)
    alert(`Error: ${title}\n${options?.description || ''}`)
  },
  info: (title: string, options?: { description?: string }) => {
    console.log('INFO:', title, options?.description)
    alert(`Info: ${title}\n${options?.description || ''}`)
  }
}

// Simple UI Components
const Card = ({ className = '', children }: { className?: string, children: React.ReactNode }) => (
  <div className={`rounded-lg border bg-white shadow-sm ${className}`}>{children}</div>
)

const CardHeader = ({ children }: { children: React.ReactNode }) => (
  <div className="flex flex-col space-y-1.5 p-6">{children}</div>
)

const CardTitle = ({ children }: { children: React.ReactNode }) => (
  <h3 className="text-2xl font-semibold">{children}</h3>
)

const CardDescription = ({ children }: { children: React.ReactNode }) => (
  <p className="text-sm text-gray-500">{children}</p>
)

const CardContent = ({ className = '', children }: { className?: string, children: React.ReactNode }) => (
  <div className={`p-6 pt-0 ${className}`}>{children}</div>
)

const CardFooter = ({ className = '', children }: { className?: string, children: React.ReactNode }) => (
  <div className={`flex items-center p-6 pt-0 ${className}`}>{children}</div>
)

const Button = ({ 
  type = 'button', 
  className = '', 
  disabled = false, 
  children, 
  onClick 
}: { 
  type?: 'button' | 'submit' | 'reset'
  className?: string
  disabled?: boolean
  children: React.ReactNode
  onClick?: () => void
}) => (
  <button
    type={type}
    className={`inline-flex items-center justify-center rounded-md bg-blue-600 px-4 py-2 text-white hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2 disabled:opacity-50 ${className}`}
    disabled={disabled}
    onClick={onClick}
  >
    {children}
  </button>
)

const Input = ({ 
  id, 
  type = 'text', 
  className = '', 
  placeholder = '', 
  value, 
  onChange, 
  disabled = false 
}: { 
  id: string
  type?: string
  className?: string
  placeholder?: string
  value: string
  onChange: (e: React.ChangeEvent<HTMLInputElement>) => void
  disabled?: boolean
}) => (
  <input
    id={id}
    type={type}
    className={`flex h-10 w-full rounded-md border border-gray-300 bg-background px-3 py-2 text-sm focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2 disabled:opacity-50 ${className}`}
    placeholder={placeholder}
    value={value}
    onChange={onChange}
    disabled={disabled}
  />
)

const Label = ({ htmlFor, children }: { htmlFor: string, children: React.ReactNode }) => (
  <label htmlFor={htmlFor} className="text-sm font-medium">
    {children}
  </label>
)

// Icons
const MailIcon = () => (
  <svg
    xmlns="http://www.w3.org/2000/svg"
    width="24"
    height="24"
    viewBox="0 0 24 24"
    fill="none"
    stroke="currentColor"
    strokeWidth="2"
    strokeLinecap="round"
    strokeLinejoin="round"
    className="absolute left-3 top-2.5 h-4 w-4 text-gray-400"
  >
    <rect width="20" height="16" x="2" y="4" rx="2" />
    <path d="m22 7-8.97 5.7a1.94 1.94 0 0 1-2.06 0L2 7" />
  </svg>
)

const LockIcon = () => (
  <svg
    xmlns="http://www.w3.org/2000/svg"
    width="24"
    height="24"
    viewBox="0 0 24 24"
    fill="none"
    stroke="currentColor"
    strokeWidth="2"
    strokeLinecap="round"
    strokeLinejoin="round"
    className="absolute left-3 top-2.5 h-4 w-4 text-gray-400"
  >
    <rect width="18" height="11" x="3" y="11" rx="2" ry="2" />
    <path d="M7 11V7a5 5 0 0 1 10 0v4" />
  </svg>
)

const APP_NAME = 'Global Health Guardian'

const LoginPage = () => {
  const [email, setEmail] = useState('')
  const [password, setPassword] = useState('')
  const [errorMessage, setErrorMessage] = useState<string | null>(null)
  const [isLoading, setIsLoading] = useState(false)
  const router = useRouter()
  const { status } = useSession()
  
  // Check if user is already logged in
  useEffect(() => {
    if (status === "authenticated") {
      router.push('/dashboard')
    }
  }, [status, router])

  const handleLogin = async (e: React.FormEvent) => {
    e.preventDefault()
    
    if (!email || !password) {
      setErrorMessage('Please enter both email and password')
      return
    }

    setIsLoading(true)
    setErrorMessage(null)
    
    try {
      const result = await signIn('credentials', {
        email,
        password,
        redirect: false
      })
      
      if (result?.error) {
        // Handle specific error messages from authentication
        console.error('Authentication error:', result.error)
        setErrorMessage('Invalid email or password. Please try again.')
        setIsLoading(false)
      } else {
        // Success - redirect
        console.log('Login successful, redirecting...')
        router.push('/dashboard')
      }
    } catch (error) {
      console.error('Login error:', error)
      setErrorMessage('An error occurred during login. Please try again.')
      setIsLoading(false)
    }
  }

  // If you encounter issues with the above approach, uncomment this fallback option
  const handleDirectLogin = async () => {
    try {
      // Use the custom API route as a fallback
      const response = await fetch('/api/test-login');
      if (response.redirected) {
        window.location.href = response.url;
      } else {
        console.error('Direct login failed, no redirect received');
        setErrorMessage('Direct login failed. Please try again or contact support.');
        setIsLoading(false);
      }
    } catch (error) {
      console.error('Direct login error:', error);
      setErrorMessage('An error occurred during direct login.');
      setIsLoading(false);
    }
  }

  return (
    <div className="flex flex-col items-center justify-center min-h-screen px-4">
      <div className="w-full max-w-md">
        <div className="text-center mb-8">
          <h1 className="text-2xl font-bold">{APP_NAME}</h1>
          <p className="text-gray-600">Multi-Tenant Healthcare Audit Platform</p>
        </div>
        
        <Card>
          <CardHeader>
            <CardTitle>Login</CardTitle>
            <CardDescription>
              Enter your credentials to access your dashboard
            </CardDescription>
          </CardHeader>
          
          <form onSubmit={handleLogin}>
            <CardContent className="space-y-4">
              {errorMessage && (
                <div className="p-3 text-sm text-red-500 bg-red-50 rounded-md">
                  {errorMessage}
                </div>
              )}

              <div className="space-y-2">
                <Label htmlFor="email">Email</Label>
                <div className="relative">
                  <MailIcon />
                  <Input 
                    id="email"
                    type="email"
                    placeholder="name@company.com"
                    className="pl-10"
                    value={email}
                    onChange={(e) => setEmail(e.target.value)}
                    disabled={isLoading}
                  />
                </div>
              </div>
              
              <div className="space-y-2">
                <div className="flex items-center justify-between">
                  <Label htmlFor="password">Password</Label>
                  <a 
                    href="#reset-password" 
                    className="text-sm text-blue-600 hover:underline"
                    onClick={(e) => {
                      e.preventDefault()
                      setErrorMessage('Password reset functionality would be implemented here.')
                    }}
                  >
                    Forgot password?
                  </a>
                </div>
                <div className="relative">
                  <LockIcon />
                  <Input 
                    id="password"
                    type="password" 
                    className="pl-10"
                    value={password}
                    onChange={(e) => setPassword(e.target.value)}
                    disabled={isLoading}
                  />
                </div>
              </div>
            </CardContent>
            
            <CardFooter className="flex-col items-stretch gap-2">
              <Button 
                type="submit" 
                className="w-full"
                disabled={isLoading}
              >
                {isLoading ? 'Signing in...' : 'Sign in'}
              </Button>
              
              <div className="text-center mt-2">
                <a 
                  href="/direct-login"
                  className="text-sm text-blue-600 hover:underline"
                >
                  Use direct login instead
                </a>
              </div>
            </CardFooter>
          </form>
        </Card>
        
        <div className="mt-4 text-center text-sm text-gray-600">
          <p>Demo credentials:<br/>
            superadmin@healthguardian.com / admin123<br/>
            hospital@healthguardian.com / hospital123
          </p>
        </div>
      </div>
    </div>
  )
}

export default LoginPage
